############################################################################################
## package 'secr'
## methods.R
## Methods for classes traps, capthist and mask
## Last changed 
## 2009 09 20 verify added to secr.fit
## 2009 09 18 investigated 'usage' summary issues
## 2009 08 24 streamlined secr.fit call string 
## 2009 08 19 model.average removed to separate file;
## 2009 09 26 local environment for 'iter'
## 2009 09 26 print.mask now works for list of masks
## 2009 09 26 sortrows argument for make.capthist
## 2009 10 02 sortrows argument for print.capthist
## 2009 10 05 split.traps
## 2009 10 07 mark resight details$q
## 2009 10 xx cellprob option in secr.fit
## 2009 10 19 fixedbeta

## This version incorporating signal strength and count detectors (but incomplete)
############################################################################################

# Following code may be used at some stage during debugging, 
# not for production version
# source ('d:\\density secr 1.2\\secr\\R\\methods.R')	
 source ('d:\\density secr 1.2\\secr\\R\\autoini.R')
 source ('d:\\density secr 1.2\\secr\\R\\functions.R')
 source ('d:\\density secr 1.2\\secr\\R\\D.designdata.R')
 source ('d:\\density secr 1.2\\secr\\R\\logmultinom.R')
 source ('d:\\density secr 1.2\\secr\\R\\secr.make.newdata.R')
 source ('d:\\density secr 1.2\\secr\\R\\secr.design.MS.R')
 source ('d:\\density secr 1.2\\secr\\R\\plot.secr.R')
 source ('d:\\density secr 1.2\\secr\\R\\sim.capthist.R')
 source ('d:\\density secr 1.2\\secr\\R\\esa.R')
 source ('d:\\density secr 1.2\\secr\\R\\derivedMS.R')	
 source ('d:\\density secr 1.2\\secr\\R\\score.test.R')	
 source ('d:\\density secr 1.2\\secr\\R\\pdot.R')	
 source ('d:\\density secr 1.2\\secr\\R\\RPSV.R')	
 source ('d:\\density secr 1.2\\secr\\R\\dbar.R')	
 source ('d:\\density secr 1.2\\secr\\R\\model.average.R')	
 source ('d:\\density secr 1.2\\secr\\R\\deviance.R')
 source ('d:\\density secr 1.2\\secr\\R\\sim.secr.R')	
 source ('d:\\density secr 1.2\\secr\\R\\confint.secr.R')	
 source ('d:\\density secr 1.2\\secr\\R\\verify.R')
 source ('d:\\density secr 1.2\\secr\\R\\secr.cellprob.R')	
#
############################################################################################
# Global variables in namespace
#
validdetectors <- c('single','multi','proximity','count','signal','areasearch')

## define a local environment (in namespace?) for temporary variables e.g. iter
## see e.g. Roger Peng https://stat.ethz.ch/pipermail/r-devel/2009-March/052883.html

.localstuff <- new.env()
.localstuff$iter <- 0

############################################################################################
# Generic methods for extracting attributes etc

usage      <- function (object, ...) UseMethod("usage")
covariates <- function (object, ...) UseMethod("covariates")
traps      <- function (object, ...) UseMethod("traps")
detector   <- function (object, ...) UseMethod("detector")
session    <- function (object, ...) UseMethod("session")
trim       <- function (object, drop, keep) UseMethod("trim")

reduce     <- function (object, columns, ...) UseMethod("reduce")
rotate     <- function (object, degrees, centrexy=NULL, ...) UseMethod("rotate")
shift      <- function (object, shiftxy, ...) UseMethod("shift")
flip       <- function (object, lr=F, tb=F, ...) UseMethod("flip")

# Default methods for specialised functions

usage.default <- function (object, ...)       {
    if (inherits(object, 'list')) lapply(object, usage.default, ...)
    else attr(object,'usage') }

covariates.default <- function (object, ...)  { 
    if (inherits(object, 'list')) lapply(object, covariates.default, ...)
    else attr(object,'covariates') }

traps.default <- function (object, ...)       {
    if (inherits(object, 'list')) {
        temp <- lapply(object, traps.default, ...)
        class(temp) <- c('list', 'traps')
        temp
    }
    else attr(object,'traps') }

detector.default <- function (object, ...)    {  
    if (inherits(object, 'list')) {
        detector.default (object[[1]], ...)
    }
    else attr(object,'detector') }

session.default <- function (object, ...)     {
    if (inherits(object, 'list')) unlist(lapply(object, session.default, ...))
    else {
        temp <- attr(object,'session') 
        names(temp) <- NULL
        temp
    }
}

trim.default <- function (object, drop, keep)     {
## drop unwanted named components of a list
## conservative resolution of conflicts between drop & keep
    objnames <- names(object)
    indices <- 1:length(object)
    if (missing(drop)) drop <- indices  # all! but wait...
    if (missing(keep)) keep <- 0
    if (is.character(keep)) keep <- match(keep, objnames)
    if (is.character(drop)) drop <- match(drop, objnames)
    drop <- drop[drop %in% indices[! (indices %in% keep)]]
    ## by index, so have to work from end
    for (i in sort(drop,decr=T)) object[[i]] <- NULL
    object
}

reduce.default <- function (object, columns, ...) { 
  object <- as.matrix(object)
  if (any(is.na(object))) warning ('NAs in input to reduce.default converted to zero')
  firsttrap <- function (y) y[abs(y)>0][1]    # first non-zero
  fnmulti   <- function (occ) apply (object[,occ,drop=F], 1, firsttrap)
  nrow <- nrow(object)
  nnew <- length(columns)
  temp <- sapply (columns, fnmulti)
  temp[is.na(temp)] <- 0
  temp
}
############################################################################################

rotate.default <- function (object, degrees, centrexy=NULL, ...) {

    rotatefn <- function (xy) {
        # about centre
        x <- xy[1] - centrexy[1]
        y <- xy[2] - centrexy[2]
        x2 <- x * cos(theta) + y * sin(theta) + centrexy[1]
        y2 <- - x * sin(theta) + y * cos(theta) + centrexy[2]
        c(x2,y2)
      }

##    if (inherits(object, 'list')) lapply(object, rotate.default, degrees=degrees, centrexy=centrexy, ...)
##    else 

    object <- as.matrix(object)

    if (abs(degrees)>0) {
        if (is.null(centrexy)) centrexy <- c(0,0)
        theta <- 2*pi*degrees/360 # convert to radians
    
        t(apply (object,1,rotatefn))
    } else object[,1:2]
}
############################################################################################

shift.default <- function (object, shiftxy, ...) { 
##    if (inherits(object, 'list')) lapply(object, shift.default, shiftxy=shiftxy, ...)
##    else 

    object <- as.matrix(object[,1:2])
    object[,1] <- object[,1] + shiftxy[1]
    object[,2] <- object[,2] + shiftxy[2]
    object
}
############################################################################################

flip.default <- function (object, lr=F, tb=F, ...) { 
##    if (inherits(object, 'list')) lapply(object, flip.default, lr=lr, tb=tb, ...)
##    else 
    object <- as.matrix(object[,1:2])
    if (is.logical(lr)) {
        if (lr) object[,1] <- 2 * mean(object[,1]) - object[,1]  ## flip about mean
    } else
        if (is.numeric(lr)) object[,1] <- 2*lr-object[,1]  ## flip about lr

    if (is.logical(tb)) {
        if (tb) object[,2] <- 2 * mean(object[,2]) - object[,2]  ## flip about mean
    } else
        if (is.numeric(tb)) object[,2] <- 2*tb-object[,2]  ## flip about tb

    object
}
############################################################################################

# Generic methods for replacing values

'usage<-' <- function (object, value) structure (object, usage = value)

'covariates<-' <- function (object, value) {
    if (!is.null(value)) {
        if (is.null(dim(value))) {
            if (length(value) != nrow(object)) stop ('length of covariate does not match', call.=F)
        }
        else 
            if (nrow(value) != nrow(object)) stop ('dimension of covariates does not match', call.=F)
        structure (object, covariates = data.frame(value))
    }
    else  
        structure (object, covariates = NULL)

}

'detector<-' <- function (object, value) {
    if (!(value %in% validdetectors)) stop ('invalid detector type')
    structure (object, detector = value)
}

'traps<-' <- function (object, value) {
    if (!is(value,'traps')) stop ("'traps' object required for replacement")

    structure (object, traps = value)
}

'session<-' <- function (object, value) {
    if (inherits(object, 'list')) {
       stop ('cannot use session<- to set session attribute of multi-session capthist object')
    }
    else {
        if (length(value) > 1) 
            stop ("require only one session name per session")
        structure (object, session = as.character(value))
    }
}

############################################################################################

######################################
## Class : traps
## defines an array of detectors
## detector may be 'single', 'multi', 'proximity' etc.
######################################

make.grid <- function (nx=6, ny=6, spacing=20, detector='multi', binomN=0,
    originxy=c(0,0), hollow=F, ID='alphay')

# 'count' detectors have attribute binomN to constrain the distribution
# of counts. Use binomN==0 for unbounded (e.g., Poisson) counts, and binomN for counts
# that cannot exceed binomN

{
    if (!( detector %in% validdetectors )) stop ('invalid detector type')

    grid <- expand.grid (x=(0:(nx-1))*spacing + originxy[1], y=(0:(ny-1))*spacing + originxy[2])

    if (! ID %in% c('numx','numy', 'numxb', 'numyb', 'alphax','alphay'))
        stop ('ID should be one of numx, numy, numxb, numyb, alphax, alphay')

if (ID %in% c('alphax','alphay')) {
  n <- max(nx,ny)/25 + 1
  ll <- sapply(1:n, function(x) apply(matrix(rep(LETTERS,rep(x,26)), nr=x),2,paste, collapse=''))
}

    if (ID == 'numy') temp <- 1:nrow(grid)
    if (ID == 'numx') temp <- t(matrix(1:nrow(grid), nc=nx))

    if (ID == 'numyb') {
        temp <- matrix(1:(nx*ny), nc=ny)
        for (i in seq(2,ny,2)) temp[,i] <- rev(temp[,i])
    }

    if (ID == 'numxb') {  ## YES
        temp <- t(matrix(1:(nx*ny), nc=nx))
        for (i in seq(2,nx,2)) temp[i,] <- rev(temp[i,])
    }

    if (ID == 'alphax') {
        colA <- ll[1:nx]
        rowA <- 1:ny
        row.names(grid) <- apply(expand.grid(colA,rowA), 1, paste, sep='', collapse='')
    }
    if (ID == 'alphay') {
        colA <- 1:nx
        rowA <- ll[1:ny]
        row.names(grid) <- apply(expand.grid(colA, rowA), 1, function(x) paste(rev(x), sep='', collapse=''))
    }

    if (hollow) {
        grid <- grid[grid$x==originxy[1] |
                             grid$x==(originxy[1] + spacing*(nx-1)) |
                             grid$y==originxy[2] |
                             grid$y==(originxy[2] + spacing*(ny-1)),]

        ## number clockwise from bottom left
        grid <- grid[order(grid$x, grid$y),]
        temp <- c(1:ny,
            t(matrix(c(
                rev ((2*ny+nx-1):(2*ny+2*nx-4)),
                (ny+1):(ny+nx-2)
              ), nc=2)),
            rev((ny+nx-1):(2*ny+nx-2)))
    }

    if (hollow | (ID %in% c('numx','numy','numxb','numyb'))) {
        row.names(grid) <- temp
        grid <- grid[order(temp),]
    }    

    attr(grid,'detector')    <- detector
    if (detector == 'count') attr(grid,'binomN') <- binomN
    attr(grid,'class')       <- c('traps', 'data.frame')
    attr(grid,'spacing')     <- spacing
    attr(grid, 'usage')      <- NULL
    attr(grid, 'covariates') <- NULL

    grid
}
############################################################################################

make.circle <- function (n = 20, radius = 100, spacing = NULL,
    detector='multi', originxy=c(0,0), IDclockwise = T)
{
    if (!( detector %in% validdetectors )) stop ('invalid detector type')

    if (is.null(radius) & is.null(spacing)) stop ('Specify one of radius or spacing')

    theta <- seq (0, 2 * pi * (n-1)/n, 2 * pi / n)
    if (!is.null(spacing)) radius <- spacing / 2 / sin(pi / n)  ## override

    object <- data.frame (x = radius * cos(theta) + originxy[1],
                          y = radius * sin(theta) + originxy[2])

    if (IDclockwise) sequence  <- c(1, n:2)
    else sequence <- 1:n
    row.names(object) <- sequence

    object <- object[order(sequence),]    
    attr(object,'detector')    <- detector
    attr(object,'class')       <- c('traps', 'data.frame')
    attr(object, 'usage')      <- NULL
    attr(object, 'covariates') <- NULL
    object
}
############################################################################################

rotate.traps <- function (object, degrees, centrexy=NULL, ...)
{
##    if (inherits(object, 'list')) lapply(object, rotate.traps, degrees, centrexy, ...)
##    else 

  rotatefn <- function (xy) {
    # about centre
    x <- xy[1] - centrexy[1]
    y <- xy[2] - centrexy[2]
    x2 <- x * cos(theta) + y * sin(theta) + centrexy[1]
    y2 <- - x * sin(theta) + y * cos(theta) + centrexy[2]
    c(x2,y2)
  }

  if (abs(degrees)>0) {
    if (is.null(centrexy)) centrexy <- c(mean(object$x), mean(object$y))
    theta <- 2*pi*degrees/360 # convert to radians

    traps2 <- data.frame(t(apply (object,1,rotatefn)))
    names(traps2) <- c('x','y')
    attr(traps2,'class')  <- c('traps', 'data.frame')
    detector(traps2)      <- detector(object)
    if (!is.null(usage(object)))
        usage(traps2)         <- usage(object)
    if (!is.null(covariates(object)))
        covariates(traps2)    <- covariates(object)
  }
  else traps2 <- object
  traps2
}
############################################################################################

shift.traps <- function (object, shiftxy, ...)
{
##    if (inherits(object, 'list')) lapply(object, shift.traps, shiftxy, ...)
##    else 

  object$x <- object$x + shiftxy[1]
  object$y <- object$y + shiftxy[2]
  object
}
############################################################################################

flip.traps <- function (object, lr=F, tb=F, ...) { 

##    if (inherits(object, 'list')) lapply(object, flip.traps, lr, tb, ...)
##    else 

    if (is.logical(lr)) {
        if (lr) object$x <- 2 * mean(object$x) - object$x  ## flip about centre
    } else
        if (is.numeric(lr)) object$x <- 2*lr - object$x  ## flip about lr

    if (is.logical(tb)) {
        if (tb) object$y <- 2 * mean(object$y) - object$y  ## flip about centre
    } else
        if (is.numeric(tb)) object$y <- 2*tb - object$y  ## flip about tb

    object
}
############################################################################################

read.traps <- function (file=NULL, data=NULL, detector='multi', ...)
{

    # count.fields(file, sep = "", quote = "\"'", skip = 0, blank.lines.skip = TRUE, comment.char = "#")

    if (!( detector %in% validdetectors )) stop ('invalid detector type')

    if (is.null(file) & is.null(data)) stop ('Must provide one of file or data')
    if (!is.null(file)) {
        nfld <- count.fields (file, ...)
        if (min(nfld) < 3) stop ('Require 3 fields (detectorID, x, y)')
        if (min(nfld) == 3) colcl <- c('character',NA,NA)
        else colcl <- c('character',NA,NA,'character')
        temp <- read.table (file, row.names=1, as.is=T, colClasses=colcl, ...)
        traps <- temp[,1:2]
        dimnames(traps)[[2]] <- c('x','y')
    }
    else traps <- data[,c('x','y')]

    detector(traps) <- detector
    class(traps)    <- c('traps', 'data.frame')

    # possible ancillary data : daily usage and covariates
    usage(traps)      <- NULL
    covariates(traps) <- NULL

    if (!is.null(file)) {
        if (ncol(temp)>2) {
            if (ncol(temp)>3) temp2 <- apply(temp[,3:ncol(temp)],1,paste)
            else temp2 <- temp[,3] 
            
            splitfield <- matrix(unlist(strsplit(as.character(temp2),'/')),byrow=T, nr=nrow(traps))   # before '/'
            used <- gsub(' ','', splitfield[,1])     # remove blanks
            used <- gsub('//t','', used)   # remove tabs
            nocc <- max(nchar(used))
            if (any(nchar(used) != nocc)) stop ('Usage fields suggest varying number of occasions')
            if (nocc>0) {
                usage(traps) <- matrix(unlist(strsplit(used,split='')), byrow=T, nc=nocc)>0
                dimnames(attr(traps, 'usage')) <- list(dimnames(traps)[[1]], 1:nocc)
            }
    
            if (ncol(splitfield)>1) {
                temp <- splitfield[,2,drop=F]
                covariates(traps) <- data.frame(C1=as.numeric(temp))
            } 
        }
    }
    traps
}
############################################################################################

print.traps <- function(x, ...) {
    if (inherits(x, 'list')) {
        for (i in 1:length(x)) {
            cat('\n')
            if (!is.null(names(x)))
                cat(names(x)[i], '\n') 
            print (x[[i]], ...)
        }
        invisible()
    }
    else {
        print(data.frame(row.names=attr(x,'row.names'), x=x$x, y=x$y), ...)
    }
}
############################################################################################

subset.traps <- function (x, subset, ...) {
    # subset may be numeric index or logical

    if (inherits(x, 'list')) {
        temp <- lapply(x, subset.traps, subset=subset, ...)
        class(temp) <- c('list', 'traps')
        temp
    }
    else {
        temp <- x[subset,,drop=F] 
        class(temp) <- c('traps','data.frame')
        detector(temp) <- detector(x)
        usage(temp) <- usage(x)   ## does not allow for subsetting of occasions 2009 09 18
        if (!is.null(covariates(x))) covariates(temp) <- covariates(x)[subset,,drop=F]
        temp
    }
}
############################################################################################

split.traps <- function (x, f, drop = FALSE, prefix='S', ...) {
  if (!inherits(x, 'traps')) stop ('argument to split.traps does not have class traps')
  if (inherits(x, 'list')) stop ('split not suitable for multi-session traps')
  options(warn=-1)
  f <- factor(f)
  if (any(!is.na(as.numeric(levels(f))))) f <- factor(paste (prefix,f,sep=''))
  options(warn=0)
  out <- list()
  for (i in levels(f)) {
    temp <- subset (x, subset = f == i, ...)
    if (!drop | (nrow(temp)>0))
      out[[i]] <- temp
  }
  class (out) <- c('list', 'traps')
  out
}

############################################################################################

rbind.traps <- function (..., renumber = TRUE) {
# combine 2 or more traps objects
  allargs <- list(...)
  check <- function (x) {

      if (!is(x,'traps')) 
          stop ('All arguments must be traps objects', call.=F)

      if (is.null(covariates(x)) != is.null(covariates(allargs[[1]]) ))
          stop ('Covariates must be provided for all or none', call.=F)

      if (!is.null(usage(x)))
          if (ncol(usage(x)) != ncol(usage(allargs[[1]]))) 
              stop ('Usage defined for varying numbers of occasions', call.=F)

      if (detector(x) != detector(allargs[[1]])) 
          warning ('Detector types vary; using first')
  }
  sapply (allargs, check)

  temp <- rbind.data.frame(...)
  class(temp) <- c('traps', 'data.frame')
  detector(temp) <- detector(allargs[[1]])

  cov <- covariates(allargs[[1]])
  if (!is.null(cov)) {
      for (i in 2:length(allargs)) cov <- rbind(cov, covariates(allargs[[i]]))
  }

  usage <- usage(allargs[[1]])
  if (!is.null(usage))
  for (i in 2:length(allargs)) usage <- rbind(usage, usage(allargs[[i]]))
  usage(temp) <- usage

  tn <- sapply(allargs, row.names, simplify=F)

  if (length(unique(unlist(tn))) != length(unlist(tn))) # renumber
  {
      if (renumber) row.names(temp) <- 1:nrow(temp)
      else {
          for (i in 1:length(tn)) tn[[i]] <- paste(tn[[i]],i,sep='.')
          row.names(temp) <- unlist(tn)
      } 
  }

  if (!is.null(usage(temp))) row.names(usage(temp)) <- unlist(tn)

  if (!is.null(covariates(temp))) {
      if (nrow(covariates(temp))>0)
      row.names(covariates(temp)) <- unlist(tn)
  }

  temp

}
############################################################################################

plot.traps <- function(x, 
    border = 100, 
    label = FALSE, 
    offset=c(6,6), 
    add = FALSE,
    hidetr = FALSE,
    detpar=list(col='red', pch=3, cex=0.8), 
    txtpar=list(col='blue', cex=0.7), 
    bg='white',
    gridlines=TRUE, 
    gridspace=100, 
    gridcol='grey',
    markvarying=FALSE,
    ... )
{
    if (inherits(x, 'list')) lapply(x, plot.traps, border, label, offset, 
        add, hidetr, detpar, txtpar, bg, gridlines, gridspace, gridcol, markvarying, ...)
    else {

        buff <- c(-border,+border)   
    
        if (!is.null(usage(x)))
            constant <- apply(attr(x,'usage'),1,function(z) !any(z>0))
        else 
            constant <- rep(TRUE,nrow(x))
        initialpar <- par(detpar)  
    
        if (!add) {
            par(bg=bg)
            eqscplot (x$x, x$y, xlim=range(x$x)+buff, ylim=range(x$y)+buff, 
                xlab='', ylab='', type='n', axes=F, ...)
    
            if (gridlines) {
                xl <- range(x$x)+buff
                yl <- range(x$y)+buff
                strtx <- gridspace * trunc(xl[1]/gridspace)
                strty <- gridspace * trunc(yl[1]/gridspace)
                finx  <- gridspace * trunc(xl[2]/gridspace+0.999)
                finy  <- gridspace * trunc(yl[2]/gridspace+0.999)
##                for (xi in seq(strtx, finx, gridspace)) segments(xi,yl[1],xi,yl[2], col=gridcol)
##                for (yi in seq(strty, finy, gridspace)) segments(xl[1],yi,xl[2],yi, col=gridcol)
## altered 2009 10 02
                for (xi in seq(strtx, finx, gridspace)) segments(xi, strty, xi, finy, col=gridcol)
                for (yi in seq(strty, finy, gridspace)) segments(strtx, yi, finx, yi, col=gridcol)

            }
        }
    
        if (hidetr==F) {
            points (x$x, x$y)
            if (markvarying & any(!constant)) {
                points (x$x[!constant], x$y[!constant], pch=16,cex=0.8)
            }
            par(txtpar)
            offsety <- ifelse (length(offset)==2, offset[2], offset[1]) 
            if (label) text (x$x+offset[1], x$y+offsety, rownames(x))
            par(initialpar)   # restore
        }
        invisible()
    }
}
############################################################################################

summary.traps <- function(object, getspacing = TRUE, ...) {

    if (inherits(object, 'list')) lapply(object, summary.traps, getspacing = getspacing, ...)
    else {

        if (is.null(object$x) | is.null(object$y)) stop ('Not a valid traps')
        nd <- length(object$x)
        if (length(object$x) != length(object$y)) stop  ('Not a valid traps')

        if (getspacing) {
            spacing <- as.matrix(dist(cbind(object$x,object$y)))
            sp <- apply(spacing,1,function(x) min(x[x>0]))
        }
        else {
            sp <- attr(object, 'spacing') 
            warning ('spacing not computed: using spacing attribute in traps object')
        }
      
        if (is.factor(covariates(object))) {
            susage <- by(usage(object), covariates(object), function(y) apply(y,2,sum))
            sumusage <- matrix(unlist(susage),byrow=T,nr=length(susage))
            dimnames(sumusage) <- list(levels(covariates(object)), names(susage[[1]]))
        }
        else if (!is.null(usage(object)))  sumusage <- apply(usage(object),2,sum)
             else sumusage <- NULL

        sumcovar <- NULL

        tempcovar <- covariates(object)  
        if (!is.null(tempcovar)) 
            if ((nrow(tempcovar)>0) & (ncol(tempcovar)>0)) ## amended to check ncol 2009 09 18
                sumcovar <- summary(tempcovar, ...)
      
        temp <- list (
          detector = attr(object,'detector'),
          ndetector = nd, 
          xrange = range(object$x), 
          yrange = range(object$y),
          spacing = mean(sp, na.rm=T),
          usage = sumusage,
          covar = sumcovar
        )

        class(temp) <- 'summary.traps'
        temp
    }
}
############################################################################################

print.summary.traps <- function (x, ...) {
  cat ('Object class    ', 'traps', '\n')
  cat ('Detector type   ', x$detector, '\n')
  cat ('Detector number ', x$ndetector, '\n')
  cat ('Average spacing ', x$spacing, '\n')
  cat ('x-range         ', x$xrange, '\n')
  cat ('y-range         ', x$yrange, '\n')
  cat ('\n')

  if (!is.null(x$covar)) {

      cat ('Summary of covariates', '\n')
      print(x$covar, ...)
  }

  if (!is.null(x$usage)) {
      cat ('Usage by occasion\n')
      print(x$usage, ...)
  }

}

############################################################################################

####################################
## Class : capthist
## capture data 
####################################

read.captures <- function (file, ...)
{
    read.table (file, as.is=T, ...)
}
############################################################################################

make.capthist <- function (captures, traps, fmt = 'trapID', noccasions = NULL, 
    covnames = NULL, bysession = TRUE, sortrows = TRUE, cut = NULL)

# captures is a dataframe with the structure:
# fmt = 'trapID'
#   column 1	Session
#   column 2	AnimalID
#   column 3	Occasion
#   column 4	TrapID
#   column 5    Signal    (optional)

# fmt = 'XY'
#   column 1	Session
#   column 2	AnimalID
#   column 3	Occasion
#   column 4	x
#   column 5	y
#   column 6    Signal    (optional)

{ 
    session <- captures[,1]
    sessionlevels <- unique(session)  ## retains order
    session <- factor(session, levels=sessionlevels)    
  
    MS <- bysession & ( length(sessionlevels) > 1)

    if (MS) {  # recursive call of make.capthist
        capturelist <- split (captures, session)
        nsession <- length(capturelist) 
        traplist <- inherits(traps, 'list')
        occvector <- length(noccasions)>1
        if (traplist & (length(traps) != nsession))
            stop ('traps list does not match capture sessions')
        if (occvector & (length(noccasions) != nsession))
            stop ('noccasions does not match capture sessions')

        capthist <- vector('list', nsession)
        for (i in 1:nsession) {
            if (traplist)  trps <- traps[[i]] else trps <- traps
            if (occvector) nocc <- noccasions[i] else nocc <- noccasions

            ## bug fixed 2009 10 01
            ## capthist[[i]]  <- make.capthist (capturelist[[i]], trps, fmt, 
            ##    nocc, covnames, FALSE, cut)

            capthist[[i]]  <- make.capthist (
                captures = capturelist[[i]], 
                traps = trps,
                fmt = fmt, 
                noccasions = nocc, 
                covnames = covnames, 
                bysession = bysession,
                sortrows = sortrows, 
                cut = cut)

        }
        names(capthist) <- levels(session)
        class(capthist) <- c('list','capthist')
        capthist
    }

    else ## single-session call
    {
        if (!(fmt %in% c('trapID','XY'))) stop ('Capture format not recognised')
        if (fmt!='trapID') {
          if (ncol(captures)<5) stop ('Too few columns in capture matrix')
          trapID    <- interaction(traps$x, traps$y)
          captTrap  <- match(interaction(captures[,4], captures[,5]), trapID)
          if (any(is.na(captTrap))) stop ('Failed to match some capture locations to detector sites')
        }
        else {
          captTrap <- match(captures[,4], row.names(traps))
        }
           
        #  if (bysession & ( length(levels(session)) > 1)) {
        #    captures[,2] <- interaction(session, captures[,2], drop = TRUE)
        #  }
      
        nocc      <- max(abs(captures[,3]))
        nocc      <- ifelse (is.null(noccasions), nocc, noccasions)
     
        if (is.null(detector(traps))) 
            stop ("require a detector type e.g. detector(traps) <- 'multi'")
        if (is.null(cut) & detector(traps)=='signal') 
            stop ("Missing cut (signal threshold) for signal data")
      
        wout <- NULL
        ID   <- NULL 
      
        uniqueID <- unique(captures[,2])
        captID <- as.numeric(factor(captures[,2], levels=uniqueID))
        nID    <- length(unique(captID))

        if (detector(traps) %in% c('proximity', 'areasearch')) {
            w     <- array (dim=c(nID, nocc, nrow(traps)))
            w[,,] <- 0
            w[cbind(captID, abs(captures[,3]), captTrap)] <- sign(captures[,3])
        }     
        else if (detector(traps) == 'signal') {
            if (is.null(cut)) stop ('Missing value for signal threshold')
            w     <- array (dim=c(nID, nocc, nrow(traps)))
            w[,,] <- 0
            if (fmt=='XY') 
                w[cbind(captID, abs(captures[,3]), captTrap)] <- sign(captures[,3]) * captures[,6]
            else 
                w[cbind(captID, abs(captures[,3]), captTrap)] <- sign(captures[,3]) * captures[,5]
            w[w<=cut] <- NA
        }
        else {
            w     <- matrix(nr=nID, nc=nocc)
            w[,]  <- 0

            ## adjusted 2009 08 13 to ensure first occurrence selected when more than one per occasion
            indices <- cbind(captID, abs(captures[,3]))
            values <- captTrap * sign(captures[,3])
            ord <- order (captID, 1:length(captID), decreasing=TRUE)
            w[indices[ord,, drop=F]] <- values[ord]    ## drop=F is critical to ensure retains dim2
        }
      
        wout <- abind(wout, w, along=1)     
        dimnames(wout)[[1]] <- uniqueID
        dimnames(wout)[[2]] <- 1:nocc

        ## optional row sort 2009 09 26
        if (sortrows) {
            roworder <- order (uniqueID)
            if (detector(traps) %in% c('proximity', 'areasearch', 'signal'))
                wout[,,] <- wout[roworder,,]
            else
                wout[,] <- wout[roworder,]
            dimnames(wout)[[1]] <- dimnames(wout)[[1]][roworder]
        }
      
        ## code to input permanent individual covariates if these are present
        zi <- NULL
        startcol <- ifelse (fmt=='trapID', 5, 6)
        if (ncol(captures) >= startcol) 
            zi <- as.data.frame(captures[,startcol:ncol(captures), drop=F])       
      
        if (!is.null(zi)) {
            # find first match of ID with positive value for each covar
            temp <- as.data.frame(matrix(nr=length(uniqueID), nc=ncol(zi)))
            for (j in 1:ncol(zi)) {
                nonmissing <- function(x) x[!is.na(x)][1]
                tempj <- tapply(zi[,j], captures[,2], unique)
                tempj2 <- sapply(tempj, nonmissing)
                temp[,j] <- factor(tempj2)       
            }
            if (is.null(covnames)) names(temp) <- names(zi)
            else {
                if (ncol(temp) != length(covnames)) stop('Number of covariate names does not match')
                names(temp) <- covnames
            }
            if (sortrows) temp <- temp[roworder,]
            attr(wout,'covariates') <- temp 
        }
      
        attr(wout, 'traps')     <- traps
        attr(wout, 'class')     <- 'capthist'
        attr(wout, 'cut')       <- cut
      
        attr(wout, 'session')   <- as.character(captures[1,1])
        wout
    }   ## end of single-session call
}
############################################################################################

discreteN <- function (n, N) {
    tN <- trunc(N)
    if (N != tN) tN + sample (x = c(1,0), prob = c(N-tN, 1-(N-tN)), replace = T, size = n)
    else rep(tN,n)
}

sim.popn <- function (D, core, buffer = 100, model2D = 'poisson', buffertype='rect',
    covariates = list(sex=c(M=0.5,F=0.5)), Ndist = 'poisson', details = NULL, seed = NULL)
{
    ##################
    ## set random seed
    ## copied from simulate.lm
    if (!exists(".Random.seed", envir = .GlobalEnv, inherits = FALSE)) 
        runif(1)
    if (is.null(seed)) 
        RNGstate <- get(".Random.seed", envir = .GlobalEnv)
    else {
        R.seed <- get(".Random.seed", envir = .GlobalEnv)
        set.seed(seed)
        RNGstate <- structure(seed, kind = as.list(RNGkind()))
        on.exit(assign(".Random.seed", R.seed, envir = .GlobalEnv))
    }
    ##################

    if (model2D == 'IHP') {
        nr <- nrow(core) 
        if (!inherits(core, 'mask'))
            stop('for model2D=IHP, core should be a habitat mask')
        if (Ndist != 'poisson') stop ('IHP not implemented for fixed N')

        nm <- rpois(nr, D * attr(core,'area'))   ## 'area' is cell area
        N <- sum(nm)
        jitter <- matrix ((runif(2*sum(nm))-0.5) * attr(core,'spacing'), nc=2)  
        animals <- core[rep(1:nr, nm),] + jitter
        animals <- as.data.frame(animals)
        xl <- range(animals[,1])
        yl <- range(animals[,2])

    }      
    else {
        if (buffertype != 'rect') stop ('rect is only buffertype available in v1.0')
        # population in arena +/- buffer from traps
        buff <- c(-buffer,+buffer)   
        xl <- range(core$x) + buff
        yl <- range(core$y) + buff
        area <- diff(xl) * diff(yl) * 0.0001  # ha
        if (!(Ndist %in% c('poisson','fixed'))) stop ('Unrecognised Ndist - should be poisson or fixed')
        N  <- switch (Ndist, poisson = rpois(1, lambda=D * area), fixed = discreteN (1, D * area))
    
        if (model2D=='poisson') {
            animals <- data.frame (x = runif(N)*diff(xl)+xl[1], y = runif(N)*diff(yl)+yl[1])  
        }
        else if (model2D=='cluster') {
            ## Neyman-Scott distribution with wrapping
            xrange <- diff(xl)
            yrange <- diff(yl)
            if (details$mu<=0) {
                nparent <- N   ## not clustered
                offspr <- sweep(matrix(runif(2*nparent),nc=2),2,c(xrange,yrange),'*')            
            }
            else {
                 nparent <- switch (Ndist, poisson = rpois(1, lambda=D * area/details$mu), 
                     fixed = discreteN (1, D * area / details$mu))
                 N <- nparent * details$mu
                 if (nparent==0) warning ('no clusters in sim.popn')
                 parent <-  sweep(matrix(runif(2*nparent),nc=2),2,c(xrange,yrange),'*')
                  
                 offspr <- matrix(rnorm(2*N),nc=2) * details$hsigma
                 parentn <- rep(1:nparent, details$mu) 
                 offspr <- offspr + parent[parentn,]
                 while (any ((offspr[,1]<0) | (offspr[,1]>xrange) | (offspr[,2]<0) | (offspr[,2]>yrange))) {
                   offspr[,1] <- ifelse (offspr[,1]<0, offspr[,1]+xrange, offspr[,1])
                   offspr[,1] <- ifelse (offspr[,1]>xrange, offspr[,1]-xrange, offspr[,1])
                   offspr[,2] <- ifelse (offspr[,2]<0, offspr[,2]+yrange, offspr[,2])
                   offspr[,2] <- ifelse (offspr[,2]>yrange, offspr[,2]-yrange, offspr[,2])
                 }
            }
            animals <- as.data.frame(sweep(offspr,2,c(xl[1],yl[1]),'+'))
        }
        else stop ('unrecognised 2-D distribution')
    }

    names(animals) <- c('x','y')
   
    attr(animals,'covariates') <- NULL
    if (!is.null(covariates)) {
        tempcov <- list()
        for (i in 1:length(covariates)) {
           covi <- sample (names(covariates[[i]]), replace=T, size=N, prob=covariates[[i]])
           temptxt <- paste ('tempcov$', names(covariates[i]), '<- covi', sep='')
           eval(parse(text=temptxt))
        }
        attr(animals,'covariates') <- as.data.frame(tempcov)
    }
    attr(animals, 'seed') <- RNGstate   ## save random seed
    attr(animals, 'Ndist') <- Ndist
    attr(animals, 'model2D') <- model2D
    attr(animals, 'boundingbox') <- expand.grid (x=xl,y=yl)[c(1,3,4,2),]
    class(animals) <- c('popn', 'data.frame')
    animals
}

############################################################################################

plot.popn <- function (x, add = FALSE, frame = TRUE, ...) {
    vertices <- attr(x,'boundingbox')
    if (add==FALSE)
    { 
        if (frame)
            eqscplot (x$x, x$y, xlab='', ylab='', xlim=range(vertices$x), 
                ylim=range(vertices$y), type='p', axes=F, ...)
        else
            eqscplot (x$x, x$y, xlab='', ylab='', type='p', axes=F, ...)
    }
    else points (x$x, x$y, ...)
    if (frame) polygon (vertices)
}
############################################################################################
	
rbind.popn <- function (..., renumber = TRUE) {
## combine 2 or more popn objects
## ... may be a single list object

    dots <- match.call(expand.dots = FALSE)$...
    allargs <- list(...)

    names(allargs) <- lapply(dots, as.character)

    if ((length(dots)==1) & (!inherits(allargs[[1]],'popn'))) allargs <- allargs[[1]]

    if (length(allargs)==1) return(allargs[[1]])

    ## check input
    check <- function (x) {
        if (!is(x,'popn')) 
            stop ('All arguments must be popn objects', call.=F)
        if (is.null(covariates(x)) != is.null(covariates(allargs[[1]]) ))
            stop ('Covariates must be provided for all or none', call.=F)
    }
    sapply (allargs, check)
  
    ## row names
    an <- unlist(sapply(allargs, row.names, simplify=F))
    names(an) <- NULL
    if (any(duplicated(an))) # renumber
    {
        if (renumber) rn <- 1:length(an)
        else {
            for (i in 1:length(an)) an[[i]] <- paste(an[[i]],i,sep='.')
            rn <- unlist(an)
        } 
    }
    else rn <- an

    ## construct output
    animals <- data.frame(abind (allargs, along=1), row.names = rn)
    names(animals) <- c('x','y')
    class(animals) <- c('popn', 'data.frame')
    attr(animals, 'Ndist') <- 'Ndist'
    attr(animals, 'model2D') <- 'model2D'
    xl <- range(sapply(allargs, function(x) attr(x,'boundingbox')$x))
    yl <- range(sapply(allargs, function(x) attr(x,'boundingbox')$y))
    attr(animals, 'boundingbox') <- expand.grid (x=xl,y=yl)[c(1,3,4,2),]
    if (!is.null(covariates(allargs[[1]]))) {
        cov <- lapply(allargs, function(x) covariates(x))
        covariates(animals) <- data.frame(abind(cov, along=1), row.names=rn)
    }
    animals
}
############################################################################################
	
subset.capthist <- function (x, subset=NULL, occasions=NULL, traps=NULL, sessions=NULL, 
    cut=NULL, dropnull=TRUE, renumber=FALSE, ...) 
{

## x - capthist object (array with 2 or 3 dimensions)
## subset - vector (character, logical or integer) to subscript rows (dim(1)) of x
## occasions - vector (integer or logical) to subscript occasions
## traps - vector (integer or logical) to subscript rows of traps object
## sessions - vector (integer or logical) to subscript sessions

    if (inherits(x,'list')) {
        if (is.null(sessions)) sessions <- 1:length(x)
        temp <- lapply (x[sessions], subset, 
            subset = subset, 
            occasions = occasions, 
            traps = traps, 
            sessions = sessions,   ## inserted 2009 10 01 
            cut = cut, 
            dropnull = dropnull, 
            renumber = renumber, ...)
        class(temp) <- c('list', 'capthist')
        if (length(temp) == 1) temp <- temp[[1]]  ## 2009 09 25
        return(temp)
    }
    else {

        detector <- detector(traps(x))
      
        if (detector %in% c('count')) 
        stop ('No subset method yet for count detections')
      
        if (is.null(occasions)) occasions <- 1:ncol(x)
        if (is.null(traps)) traps <- 1:nrow(traps(x))
        if (is.null(subset)) subset <- 1:nrow(x) 
        if (is.null(cut)) cut <- attr(x, 'cut')
        if (is.character(subset)) subset <- dimnames(x)[[1]] %in% subset	
        else if (!is.logical(subset)) subset <- (1:nrow(x)) %in% subset
        if (!is.logical(occasions)) occasions <- (1:ncol(x)) %in% occasions
        ## condition missing values
        if (detector %in% c('signal'))
             x[is.na(x)] <- -1e20
        else
             x[is.na(x)] <- 0
      
        ## prepare to drop null histories
        if (dropnull) {
      
            if (detector %in% c('areasearch', 'proximity'))
                nonnull <- apply(abs(x[,occasions, traps, drop=F]),1,sum) > 0
            else 
            if (detector %in% c('signal'))
                nonnull <- apply(x[,occasions, traps, drop=F] > cut,1,sum) > 0
            else {
                x[!(abs(x) %in% traps)] <- 0   
                nonnull <- apply(abs(x[,occasions, drop=F]),1,sum) > 0
            }
        }
        else nonnull <- T
      
        ## perform main subset operation
        OK1 <- subset & !is.na(subset) & nonnull
      
        if (detector %in% c('proximity', 'signal'))
            temp <- x[OK1, occasions, traps, drop=F]
        else
            temp <- x[OK1, occasions, drop=F]

        if (detector == 'signal')
            temp[temp <= cut] <- NA
      
        nrow <- nrow(temp)
        nocc <- ncol(temp)  
       
        ## drop null occasions
        OK2 <- rep(T,nocc)
      
        if ((detector!='signal') &        ## no system yet for multi-occasion signal data
              any( apply(abs(temp),2,sum) ==0)) 
        {
            if (dropnull) { 
                OK2 <- apply(abs(temp),2,sum) > 0
                if (detector %in% c('proximity'))
                    temp <- temp[,OK2,,drop=F]
                else
                    temp <- matrix(temp[,OK2,drop=F], nr=nrow)
            }
            else 
                warning (paste('No detections on occasion(s) : ', paste((1:nocc)[apply(abs(temp),2,sum) ==0]),'\n  '))
            nocc <- dim(temp)[2]  # refresh
        }
      
        class(temp) <- 'capthist'
      
        ## subset permanent covariates
        if (!is.null(covariates(x)))
           covariates(temp) <- covariates(x)[subset & !is.na(subset) & nonnull,,drop=F]
      
        ## session
        session(temp) <- session(x)
  
        ## subset traps
        traps(temp) <- subset.traps (traps(x), traps)  
      
        ## subset usage
        if (!is.null(usage(traps(temp)))) {
            usage(traps(temp)) <- usage(traps(temp))[,occasions,drop=F][,OK2,drop=F]  # sequential subseting of cols
            OK3 <- apply(usage(traps(temp)), 1, sum) > 0
            usage(traps(temp)) <- usage(traps(temp))[OK3,,drop=F]
            newtrap <- 1:nrow(traps(temp))
            newtrap[OK2] <- 1:sum(OK3)
            ## drop unused traps
            traps(temp) <- subset(traps(temp), OK3)
            ## must also renumber traps in capthist
            temp[abs(temp)>0] <- sign(temp[abs(temp)>0]) * newtrap[abs(temp)]
        }
      
        ## renumber if desired
        if (renumber) {
            if (detector %in% c('signal','proximity')) 
                dimnames(temp) <- list(1:nrow,1:nocc,NULL)   # renew numbering
            else 
                dimnames(temp) <- list(1:nrow,1:nocc)  
        }

        attr(temp, 'cut') <- cut  ## but doesn't pass null value, uses cut() instead
        temp
    }
    
}
############################################################################################

MS.capthist <- function (...) {
    # make a list of capthist objects, each for one session
    dots <- match.call(expand.dots = FALSE)$...
    n <- length(dots)
    notOK <- logical(n)
    for (i in 1:n) notOK[i] <- inherits(get(as.character(dots[i])), 'list')
    if (any (notOK)) stop ('cannot combine multi-session capthist objects')
    MS <- list(...)
    names(MS) <- sapply(dots, as.character)  
    if (any(duplicated(names(MS)))) stop ('Sessions must have unique names')
    class(MS) <- c('list', 'capthist')
    for (i in 1:length(MS)) session(MS[[i]]) <- names(MS)[i]  ## force conformity
    MS
}
############################################################################################

rbind.capthist <- function (..., renumber = TRUE, pool = NULL)
## NOT S3 method (for now) because then naming lost...
## ... argument may be :

{
    dots <- match.call(expand.dots = FALSE)$...
    allargs <- list(...)
    names(allargs) <- lapply(dots, as.character)
    if (length(dots)==1) object <- allargs[[1]]
    else object <- allargs

    ## Case 1
    ## several lists or a combination of list & elementary capthist objects
    ## concatenate lists, including elementary objects (ignore 'pool')

    if ((length(dots)>1) & any(sapply(allargs, is.list))) {
        if (!is.null(pool)) warning('pool argument ignored in rbind.capthist')
        temp <- c(...)
        class (temp) <- c('list', 'capthist')
        return(temp)
    } 
    
    ## Case 2
    ## a single MS capthist (i.e. a list)
    ## rbind components as identified in 'pool'
    ## recursive call for each component of 'pool'

    if((length(dots)==1) & (is.list(object)) & !is.null(pool)) {     
        if (!is.list(pool)) {
            pool <- list(combined=1:length(object))
            warning ('list not specified, pooling all components')
        }
        else if (any (sapply(unlist(pool), function(x) length(object[[x]])==0)))
                stop('invalid pooling indices') ## prempted by 'subscript out of bounds'

        temp <- lapply (pool, function (x) {
            temphist <- object[x]
            class(temphist) <- c('list', 'capthist')
            rbind.capthist(temphist, pool=NULL) })

        if (length(temp)==1) {
            temp <- temp[[1]]
            class(temp) <- 'capthist'
        }
        else class (temp) <- c('list', 'capthist')
        names(temp) <- session(temp)

        return(temp)
    }
    else {

    ## Case 3
    ## 1 to several several elementary capthist objects
    ## conventional rbind, given compatible traps, covariates, noccasion
    ## optional renumbering
    
        check <- function (x) {
          if (!is(x,'capthist')) 
              stop ('all arguments must be capthist objects', call.=F)
          if (is.null(covariates(x)) != is.null(covariates(object[[1]]) ))
              stop ('covariates must be provided for all or none', call.=F)
          if (any(dim(x)[-1] != dim(object[[1]])[-1]))
                  stop ('varying numbers of occasions and/or traps', call.=F)
        }

        if (length(dots)==1) 
       
        sapply (object, check)
        temp <- abind(..., along=1)
        class(temp) <- c('capthist')
        traps(temp) <- traps(object[[1]])
      
        ## covariates
        tempcov <- covariates(object[[1]])
        if (!is.null(tempcov)) {
           for (i in 2:length(object))
           tempcov <- rbind(tempcov, covariates(object[[i]]))
        } 
    
        covariates(temp) <- tempcov
          session (temp) <- paste(names(object), collapse='+') ## or a shorter more descriptive title?

        if (renumber) {
            ID <- unlist(sapply(object, rownames))
            source <- rep(names(object), sapply(object, nrow))
            rownames(temp) <- paste(source, ID, sep='.') ## assign new rownames?
        }
      
        temp
    }
}
############################################################################################

#----------------------------------------------------
# Dimension	2	2	3		3
#----------------------------------------------------
# 		single	multi	proximity	count
# single	&#	&	*		*
# multi		&#	&	*		*
# proximity	&#	&	*		*
# count		&#@	&@	@		*
#----------------------------------------------------
#  * no loss of data
#  # must choose among animals (more than one animal)
#  & must choose among traps (animal caught more than once)
#  @ count reduced to presence

############################################################################################

reduce.capthist <- function (object, columns, ..., outputdetector = NULL, random = TRUE) {

# columns - list, each element gives occasions to include in new capthist

    firsttrap <- function (y) {
        y <- t(y)         # allow for occason x trap matrix in proximity, count data
                          # i.e. select trap 48 on day 3 ahead of trap 40 on day 4
        y[abs(y)>0][1]    # first non-zero
    }
    randomtrap <- function (y) {
        y <- y[abs(y)>0]
        if (length(y)<1) y <- 0
        if (length(y) == 1) y
        else sample (size=1, y)    # random non-zero, weighted by frequency in sample
    }
  
    blanktraps <- logical (nrow(traps(object)))
    alltraps  <- function (y) {
        tr <- blanktraps
        tr[y[abs(y)>0]] <- TRUE       # all non-zero
        tr 
    }
  
    #----------------------------------------------------------------------------
    # functions applied to collapse a set of occasions 'occ' to a single occasion
    # result is a vector (single, multi detectors) or
    # a matrix (proximity, signal, count detectors) 
  
    fnmulti.f <- function (occ) apply (object[,occ,drop=F], 1, firsttrap)
    fnmulti.r <- function (occ) apply (object[,occ,drop=F], 1, randomtrap)
    fnp2multi.f <- function (occ) {
        temp <- object[,occ,,drop=F]
        temp[,,] <- aperm(apply(temp, 1:2, function(x) (1:length(x)) * x), c(2,3,1))  
        apply (temp, 1, firsttrap)
    }
    fnp2multi.r <- function (occ) {
        temp <- object[,occ,,drop=F]      
        temp[,,] <- aperm(apply(temp, 1:2, function(x) (1:length(x)) * x), c(2,3,1))  
        apply (temp, 1, randomtrap)
    }
  
    fnprox    <- function (occ) apply (object[,occ,,drop=F], c(1,3), function (x) any(x>0) )
    fncount   <- function (occ) apply (object[,occ,,drop=F], c(1,3), sum)
    fnm2p     <- function (occ) t(apply (object[,occ,drop=F], 1, alltraps))
  
    #----------------------------------------------------------------------------
  
    if (inherits(object,'list')) stop ('reduce.capthist not designed for multiple sessions')
    nrow <- nrow(object)
    nnew <- length(columns)
    inputdetector <- detector(traps(object))
    if (is.null(outputdetector)) outputdetector <- inputdetector
  
    # assume validdetectors[1:4] == c('single','multi','proximity','count')
    if (!outputdetector %in% validdetectors) stop ('Unrecognised output detector type')
    if (outputdetector == 'single')
    stop ("No reduce method yet for outputdetector = 'single'")
    inputdetn <- match (inputdetector, validdetectors[1:4])
    outputdetn <- match (outputdetector, validdetectors[1:4])
    if (is.na(inputdetn) | is.na(outputdetn))
    stop ('No reduce method yet for this combination of detectors')
  
    if ((outputdetector == 'proximity') & (inputdetector == 'multi')) {
        temp <- sapply (columns, fnm2p)
        temp <- array (temp, dim=c(nrow,nrow(traps(object)),nnew))
        temp <- aperm(temp, c(1,3,2))
    }
    else if (outputdetector == 'proximity') {
        temp <- sapply (columns, fnprox)
        temp <- array (temp, dim=c(nrow,nrow(traps(object)),nnew))
        temp <- aperm(temp, c(1,3,2))
    }
    else if (outputdetector == 'count') {
        temp <- sapply (columns, fncount)
        temp <- array (temp, dim=c(nrow,nrow(traps(object)),nnew))
        temp <- aperm(temp, c(1,3,2))
    }
    else if (inputdetector %in% c('multi', 'single')) {
        if (random) temp <- sapply (columns, fnmulti.r)
        else        temp <- sapply (columns, fnmulti.f)
    }
    else if (inputdetector %in% 'proximity') {
      if (random) temp <- sapply (columns, fnp2multi.r)
      else        temp <- sapply (columns, fnp2multi.f)
    }
    else stop ('Combination of input and output detectors not implemented')

    class(temp) <- 'capthist'
    covariates(temp) <- covariates(object)
    traps(temp) <- traps(object)  # drop traps not used on occasions 
    detector(traps(temp)) <- outputdetector
  
    if (!is.null(usage(traps(temp)))) {
        usage(traps(temp)) <- usage(traps(temp))[,columns,drop=F][,OK,drop=F]  # sequential subseting of cols
        OK <- apply(usage(traps(temp)), 1, sum) > 0
        usage(traps(temp)) <- usage(traps(temp))[OK,,drop=F]
        newtrap <- 1:nrow(traps(temp))
        newtrap[OK] <- 1:sum(OK)
  
        traps(temp) <- subset(traps(temp), OK)
  
        # must also renumber traps in capthist
  
        temp[abs(temp)>0] <- sign(temp[abs(temp)>0]) * newtrap[abs(temp)]
    }
  
# deleted 2009 07 19
#    attr(temp, 'nocc') <- nnew
  
    if (outputdetector %in% c('count', 'proximity')) {
        dimnames(temp) <- list(1:nrow,1:nnew,NULL)   # renew numbering
        temp[is.na(temp)] <- F
    }
    else {
        dimnames(temp) <- list(1:nrow,1:nnew)  
        temp[is.na(temp)] <- 0
    }
    
    temp
}
############################################################################################

print.capthist <- function (x,..., condense = FALSE, sortrows = FALSE)
{  
    ## recursive if list of capthist
    if (inherits(x, 'list')) lapply (x, print.capthist, ..., condense = condense, 
        sortrows = sortrows)
    else { # strip attributes, but why bother?
        cat('Session = ', session(x), '\n') 
        if (condense & (detector(traps(x)) %in% c('proximity'))) {
            temp <- apply(x, 3, function(y) y[apply(abs(y),1,sum)>0,, drop=F])
            trapnames <- rownames(traps(x))
            traps <- trapnames[rep(1:length(temp), sapply(temp, nrow))] 
            detections <- as.matrix(abind(temp, along=1))
            temp <- data.frame(AnimalID = rownames(detections), Trap = traps, detections, stringsAsFactors = FALSE, row.names=NULL)
            names(temp)[3:ncol(temp)] <- 1:(ncol(temp)-2)
            if (sortrows) temp <- temp[order(temp[,1]), ]
            rownames(temp) <- 1:nrow(temp)
            print(temp)
            invisible (temp)
        }
        else {
            temp <- array (x, dim=dim(x))
            dimnames(temp) <- dimnames(x)
            if (sortrows) {
                if (length(dim(x)) == 3) temp <- temp[order(dimnames(temp)[[1]]),,]
                else temp <- temp[order(dimnames(temp)[[1]]),]
            }
            print.default(temp,...)
            invisible(temp)
        }
    }
}
############################################################################################

plot.capthist <- function(x, rad = 5, hidetraps = FALSE, tracks = FALSE, 
   title = TRUE, subtitle = TRUE,
   add = FALSE,
   varycol = FALSE, icolours = NULL, randcol = FALSE, 
   lab1cap = FALSE, laboffset = 4,
   ncap = FALSE,
   splitocc = NULL, col2 = 'green',
   cappar = list(cex=1.3, pch=16, col='blue'),
   trkpar = list(col='blue', lwd=1), 
   labpar = list(cex=0.7, col='black'),
   ...)

    # suggest include optional session label 2009 02 13
    # see also version in d:\single sample with stems=F, mst=F 2009 02 22

{
    ## recursive if list of capthist
    if (inherits(x, 'list')) {
        sapply (x, plot.capthist, 
            rad = rad, hidetraps = hidetraps, tracks = tracks, 
            title = title, subtitle = subtitle, add = add, varycol = varycol, icolours = 
            icolours, randcol = randcol, lab1cap = lab1cap, laboffset = 
            laboffset, ncap = ncap, splitocc = splitocc, col2 = col2,
            cappar = cappar, trkpar = trkpar, labpar = labpar, ...)
    }
    else {

        traps <- traps(x)
        detectr <- detector(traps)
        nocc <- ncol(x)

        if (!add) plot(traps, hidetr=hidetraps, ...)
        initialpar <- par(cappar)

        if (is.null(icolours)) icolours <- terrain.colors((nrow(x)+1)*1.5)

        if (varycol) { 
            if (randcol) icolours <- sample(icolours)
            palette(icolours)
            icol <- 0 
        }

        plotprox <- function (x) {
            x <- abs(x)   # do not distinguish deads for now     
            # some dupl points will be over plotted - could increase rad for
            # captures after first at a trap on a given day
            dx  <- rep((cos((1:nocc) * 2 * pi / nocc) * rad), ncol(x))[x>0]
            dy  <- rep((sin((1:nocc) * 2 * pi / nocc) * rad), ncol(x))[x>0]
            if (varycol) icol <<- icol+1  ## 2009 10 02
            par(trkpar)
            if (varycol) par(col=icol)    ## override 2009 10 02
            if (tracks) lines (traps$x[x]+dx, traps$y[x]-dy)
            par(cappar)
            if (varycol) par(col=icol)    ## override 2009 10 02
            points (traps$x[x]+dx, traps$y[x]-dy)
        }
    
        plotcapt <- function (x) {
            x <- abs(x)   # do not distinguish deads for now
            dx <- (cos((1:nocc) * 2 * pi / nocc) * rad)
            dy <- (sin((1:nocc) * 2 * pi / nocc) * rad)
            occ2 <- (1:nocc) %in% splitocc
            x0 <- x>0
            x1 <- x0 & !(occ2)
            x2 <- x0 & occ2
    
    
            x[!x0] <- 1   # fool into keeping length..
            usesplit <- !is.null(splitocc) & (sum(x2)>0)
            if (varycol) icol <<- icol+1
            par(trkpar)
            if (varycol) par(col=icol)   # override
    
            if (tracks) {
                if (usesplit) { 
                    par(col=col2)
                    lines ((traps$x[x]+dx)[x0], (traps$y[x]-dy)[x0])  # all
                    par(trkpar)
                    if (varycol) par(col=icol)   # override
                    lines ((traps$x[x]+dx)[x1], (traps$y[x]-dy)[x1])  # pre-split
                }
                else
                lines ((traps$x[x]+dx)[x0], (traps$y[x]-dy)[x0])  # all
            }
    
            par(cappar)
            if (varycol) par(col=icol)   # override
            points ((traps$x[x]+dx)[x0], (traps$y[x]-dy)[x0])
            if (usesplit) { 
              par(col=col2)
              points ((traps$x[x]+dx)[x2], (traps$y[x]-dy)[x2])
            }
            
        }
    
        labcapt <- function (n) {
            t1 <- abs(x[n,])
            o1 <- sum(cumsum(t1)==0)+1   # first occasion
            t1 <- t1[o1]                 # first trap site
            dx  <- (cos((1:nocc) * 2 * pi / nocc) * rad)[o1]
            dy  <- (sin((1:nocc) * 2 * pi / nocc) * rad)[o1]        
            par(labpar)
            if (varycol) par(col=n)   # override
            laboffsety <- ifelse (length(laboffset)==2, laboffset[2], laboffset[1])
            text (traps$x[t1]+dx+laboffset[1], traps$y[t1]-dy+laboffsety, row.names(x)[n])
        }
    
        ncapt <- function (x) {
            if (detectr == 'proximity')
               temp <- apply (x,c(2,3),sum) # capts/trap/day
            else {
               fx   <- factor(x, levels=0:nrow(traps))
               temp <- table (fx, col(x)) # capts/trap/day
               temp <- temp[-1,,drop=F] # drop zeros
            }
            dx  <- rep(cos((1:nocc) * 2 * pi / nocc) * rad, rep(nrow(traps),nocc))
            dy  <- rep(sin((1:nocc) * 2 * pi / nocc) * rad, rep(nrow(traps),nocc))
            par(labpar)
            par(adj=0.5)
            OK <- temp>0
            text ((traps$x[row(temp)]+dx)[OK], (traps$y[row(temp)]-dy)[OK], as.character(temp[OK]))
        }
    
        plotsignal <- function (x, maxsignal) {
            # x is a matrix of animals by times for one trap
            x <- abs(x)   # do not distinguish deads for now     
            # some dupl points will be over plotted - could increase rad for
            # captures after first at a trap on a given day
    
            n <- nrow(x)
            k <<- k+1
          
            dx  <- rep( (cos((1:nrow(x)) * 2 * pi / n) * rad), ncol(x))
            dy  <- rep( (sin((1:nrow(x)) * 2 * pi / n) * rad), ncol(x))
    
            sq <- order(x)     # plot darkest points last
            sq <- sq[x[sq]>0]  # drop undetected 
            dx <- dx[sq]
            dy <- dy[sq]
            x  <- x[sq]
    
            par(cappar)
            points (traps$x[k]+dx, traps$y[k]-dy, col = grey(1-x/maxsignal))
        }

        if ( detectr %in% c('areasearch', 'proximity') ) 
        {
            w <- apply(x,1:2,function(x) (abs(x)>0) * (1:length(x)))
            w <- aperm(w, c(2,3,1))
            apply( w, 1, plotprox )
        }
        else 
        if ( detectr == 'signal' ) 
        {
            # plot one detector at a time
            k <- 0
            apply( x, 3, plotsignal, maxsignal = max(x))
        }
        else  {
            apply( x, 1, plotcapt )
        }
    
        if (lab1cap) for (i in 1:nrow(x)) labcapt(i)
        if (ncap) { ncapt(x)}
    
        par(initialpar)   # restore
    
        if (is.logical(title)) {
            txt <- ifelse (is.null(session(x)), paste(deparse(substitute(x)), collapse=''), session(x))
            title <- ifelse(title, txt, '')
        } 

        if (title != '') {
            par(col='black')
            mtext(side=3,line=1.2, text = title, cex=0.7)
        }

        if (is.logical(subtitle)) 
            subtitle <- ifelse(subtitle, paste(nocc, 'occasions,' , sum(abs(x)>0),
                'detections,', nrow(x), 'animals'), '')

        if (subtitle != '') {
            par(col='black')
            mtext(text = subtitle, side=3,line=0.2, cex=0.7)
        }

        invisible(sum(abs(x)>0))
    }
}
############################################################################################

summary.capthist <- function(object, ...) {

    ## recursive if list of capthist
    if (inherits(object, 'list')) lapply (object, summary.capthist, ...)
    else {

        traps <- traps(object)
        detector <- detector(traps)
        cut <- attr(object, 'cut')   # signal strength only
        nd <- length(traps$x)
        spacing <- as.matrix(dist(cbind(traps$x,traps$y)))
        sp <- apply(spacing,1,function(x) min(x[x>0]))
      
        # ni, ui, fi, M, losses etc.
        nocc <- ncol(object)
        counts <- matrix(nr = 8, nc = nocc)
        counts[,] <- 0
        signalsummary <- NULL
     
        if (detector == 'signal') {
            detected <- apply( object[,,,drop=F], c(1,2), function(x) any(x>cut, na.rm=T))
            if (nocc>1) stop ('Not configured for more than one occasion when detector == signal')
            else {
                counts [1,1] <- sum(detected) 
                counts [2,1] <- counts [1,1]
                counts [3,1] <- counts [1,1]
                counts [4,1] <- counts [1,1]
                counts [5,1] <- 0                ## cannot code losses with 'signal'
            }
            counts [6,] <- apply(object[,,, drop=F], 2, function(x) sum(x > cut, na.rm=T))
            tempt <- apply(object[,,,drop=F],c(2,3), function(x) sum(x > cut, na.rm=T))
            counts [7,] <- apply(tempt,1,sum)  
    
            signals <- object[,,]
            signals <- signals[ !is.na(signals) ]
            signals <- signals[ signals > cut ]
            signalsummary <- summary(signals)
        }
    
        else
##        if (detector %in% c('proximity', 'count', 'areasearch')) {
          if (length(dim(object)) > 2) {
      
          tempx <- apply( object[,,,drop=F], c(1,2), function(x) sum(abs(x))>0)
          if (nocc>1)   # distinction may not be needed... 
          {
              counts [1,] <- apply(tempx, 2, function(x) sum(abs(x)>0) )
              tempx2 <- apply(tempx, 1, function(x) cumsum(abs(x))>0)
              counts [4,] <- apply(tempx2,1,sum)
              counts [2,] <- c(counts[4,1],diff(counts[4,]))
              counts [3,] <- tabulate(apply(tempx,1, function(x) sum(abs(x)>0)),nbin=nocc)
              counts [5,] <- apply(tempx,2, function(x) sum(x<0))       ## doesn't work for signal
      
          }
          else {
              counts [1,1] <- sum(abs(tempx)>0) 
              counts [2,1] <- counts [1,1]
              counts [3,1] <- counts [1,1]
              counts [4,1] <- counts [1,1]
              counts [5,1] <- sum(tempx<0)       ## doesn't work for signal
          }
      
          if (detector %in% c('single','multi','proximity','signal','areasearch'))
             counts [6,] <- apply(object[,,, drop=F], 2, function(x) sum(abs(x)>0))
          if (detector %in% c('count'))
             counts [6,] <- apply(object[,,, drop=F], 2, function(x) sum(x))
      
          tempt <- apply(object[,,,drop=F],c(2,3), function(x) sum(x)>0)
          counts [7,] <- apply(tempt,1,sum)
      
        }
        else  {                           # detector 'single', 'multi'
    
          if (nocc == 1)
          {
              counts [1,1] <- sum(abs(object)>0) 
              counts [2,1] <- counts [1,1]
              counts [3,1] <- counts [1,1]
              counts [4,1] <- counts [1,1]
              counts [5,1] <- sum(object<0)
          }
          else {
              counts [1,] <- apply(object[,,drop=F], 2, function(x) sum(abs(x)>0))
          
              if (nrow(object) > 0) {
                  tempM <- apply(object[,,drop=F], 1, function(x) cumsum(abs(x))>0)
                  counts [4,] <- apply(tempM,1,sum)
              }
              counts [2,] <- c(counts[4,1], diff(counts[4,]))
              counts [3,] <- tabulate(apply(object[,,drop=F],1, function(x) sum(abs(x)>0)),nbin=nocc)
              counts [5,] <- apply(object[,,drop=F],2, function(x) sum(x<0))
          }
      
          counts [6,] <- apply(object[,,drop=F],2, function(x) sum(abs(x)>0))
          counts [7,] <- apply(object[,,drop=F],2, function(x) length(unique(x[x!=0])))
        }
      
        if (!is.null(usage(traps))) 
            counts[8,] <- apply(usage(traps),2,function(x) sum(x>0))
        else 
            counts [8,] <- rep(nrow(traps),nocc)
      
        counts <- as.data.frame(counts)
        dimnames(counts) <- list(c('n','u','f','M(t+1)','losses','detections','traps visited','traps set'), 1:nocc)
        counts$Total <- apply(counts, 1, sum)
        counts$Total[4] <- counts[4, nocc]
      
        if ((!(detector %in% c('proximity','count','signal','areasearch'))) & (-diff(counts$Total[1:2]) > 1)) {
           PSV <- RPSV(object)
           dbar <- dbar(object)
        }
        else { PSV <- NULL; dbar <- NULL } 
      
        temp <- list (
            detector = detector, 
            ndetector = nd, 
            xrange = range(traps$x), 
            yrange = range(traps$y),
            spacing = mean(sp, na.rm=T),
            counts = counts,
            dbar = dbar,
            RPSV = PSV,
            cut = cut,        # signal only
            signalsummary = signalsummary
        )
      
        class(temp) <- 'summary.capthist'
        temp
    }
}
############################################################################################

counts <- function (CHlist, counts = 'M(t+1)') {
    if (!inherits(CHlist, 'capthist')) stop ('require capthist object')
    getc <- function (cnt) { 
        getcnt <- function(x, maxocc) {
            temp <- x$counts[cnt,]
            lt <- length(temp)
            matrix(c(temp[-lt], rep(NA, maxocc-lt+1), temp[lt]),nr=1)
        }
        if (!is.list(CHlist))
            summary(CHlist)$counts[cnt,]
        else {
            maxocc <- max(sapply(CHlist,ncol))
            abind(lapply(summary(CHlist), getcnt, maxocc), along=1, 
                new.names=list(session(CHlist), c(1:maxocc, 'Total')))
        }
    }
    temp <- lapply (counts, getc)
    names(temp) <- counts
    temp
}

print.summary.capthist <- function (x, ...) {
    cat ('Object class     ', 'capthist', '\n')
  
    cat ('Detector type    ', x$detector, '\n')
    
    cat ('Detector number  ', x$ndetector, '\n')
    cat ('Spacing          ', x$spacing, 'm \n')
    cat ('x-range          ', x$xrange, 'm \n')
    cat ('y-range          ', x$yrange, 'm \n')
  
    if (!is.null(x$RPSV)) {
        cat ('RPSV             ', x$RPSV, 'm \n')
        cat ('dbar             ', x$dbar, 'm \n')
    }
    cat ('\n')
    cat ('Counts by occasion \n')
    print(x$counts, ...)

    cat ('\n')

    if (x$detector == 'signal') {
        cat ('Signal threshold ', x$cut, '\n')
        print (x$signalsummary)
    }  
}
############################################################################################

# datacheck.capthist <- function(x) {
# 
#     prox <- detector(traps(x))=='proximity'
#
#     # all detector locations appear in 'traps'
#     temp <- x[abs(x)>0]   # DOES NOT APPLY TO PROXIMITY DETECTORS - RE-DO
# 
#     ntrap <- nrow(attr(x,'traps'))
#     notfound <- !(temp %in% 1:ntrap)
#     if (any (notfound)) {
#         cat ('Detections at these unknown sites : ', temp[notfound], '\n')
#     }
#     else cat ('All detections were at known sites','\n')
# 
#     # trap not used on occasion s
#     if (!is.null (attr(x,'usage'))) {
#         used <- !attr(x,'usage')
#         if (prox) x <- t(apply(x, c(2,3), sum))   # collapse
#         conflict <- (abs(x)>0) & notused    ##### notused???
#         if (any(conflict)) {
#             cat('Detections at sites that were not in use','\n')
#             print(cbind(trap=row(conflict), occasion=col(conflict))[conflict])
#         }
#         else cat ('No detections at unused sites','\n')
#     }
# }
############################################################################################

###############################
## Class : mask
## defines a habitat mask
###############################

make.mask <- function (traps, buffer = 100, spacing = NULL, nx = 64, type = 'traprect',
    poly = NULL, pdotmin = 0.001, ...) 
{
    insidepoly <- function (xy) {      
        temp <- .C('inside',  PACKAGE = 'secr',
            as.double (xy),
            as.integer (np),
            as.double (poly),
            result = integer(1))
        as.logical(temp$result)
    }

    if (inherits(traps, 'list')) {         ## a list of traps objects
        if (inherits(poly, 'list')) 
            stop ('lists of polygons not implemented in make.mask')
        temp <- lapply (traps, make.mask, buffer = buffer, spacing = spacing, nx = nx, 
            type = type, poly = poly, pdotmin = pdotmin, ...)
        class (temp) <- c('list', 'mask')
        temp 
      }
    else {
  
        if (! (type %in% c('traprect','trapbuffer','polygon', 'pdot'))) 
            stop ('Mask type must be one of traprect, trapbuffer, polygon or pdot')
  
        dots <- match.call(expand.dots = FALSE)$...
        if ((length(dots)==0) & (type == 'pdot')) warning ('no detection parameters supplied for make.mask; using defaults')
  
        buff <- c(-buffer,+buffer)
  
        if (type=='polygon') {
            if (is.null(poly)) stop ('polygon must be supplied')
            poly <- matrix(unlist(poly),nc=2)
            poly <- rbind (poly, poly[1,])  # force closure of poly
            np <- nrow(poly)
            if (any (!apply(traps,1,insidepoly))) warning ('some traps are outside polygon')
            xl <- range(poly[,1])
            yl <- range(poly[,2])
        }
        else {
            xl <- range(traps$x) + buff
            yl <- range(traps$y) + buff
        }
      
        if (is.null(spacing)) spacing <- diff(xl) / nx
        mask   <- expand.grid (
                    x=seq(xl[1] + spacing/2, xl[2], spacing), 
                    y=seq(yl[1] + spacing/2, yl[2], spacing)
                  )
  
        attr(mask,'out.attrs') <- NULL   ## added 2009 07 03
      
        if (type=='trapbuffer') {
            mask <- mask[distancetotrap(mask, traps) <= buffer,]
        }
  
        if (type=='polygon') {
            mask <- mask[apply(mask,1,insidepoly),]   
            attr(mask,'polygon') <- poly   # save
        }
        else
            attr(mask,'polygon') <- NULL
      
        if (type=='pdot') {
            OK <- pdot(mask, traps = traps, ...) > pdotmin
            edge <- function (a,b) any (abs(a-b) < (spacing))
            mask <- mask[OK,]
            attr(mask,'pdotmin') <- pdotmin   # save nominal threshold
            if (edge(mask[,1],xl[1]) |
                edge(mask[,1],xl[2]) |
                edge(mask[,2],yl[1]) |
                edge(mask[,2],yl[2]))
            warning ('pdot mask may have been truncated; possibly increase buffer')
        }
    
        # get mean and SD of numeric columns
        statfn <- function(x) if (is.numeric(x)) c(mean(x), sqrt(var(x))) else c(NA,NA)
        
        attr(mask,'type')        <- type
        attr(mask,'meanSD')      <- as.data.frame (apply(mask, 2, statfn))
        attr(mask,'area')        <- spacing^2 * 0.0001
        attr(mask,'spacing')     <- spacing
        attr(mask,'boundingbox') <- expand.grid(x=xl,y=yl)[c(1,2,4,3),]
        class(mask)  <- c('mask', 'data.frame')
      
        mask
    }
}
###############################################################################

subset.mask <- function (x, subset, ...) {
    # subset may be numeric index or logical
    temp <- x[subset,,drop=F] 
    
    statfn <- function(x) if (is.numeric(x)) c(mean(x), sqrt(var(x))) else c(NA,NA)
    attr(temp,'type')        <- 'subset'
    attr(temp,'meanSD')      <- as.data.frame (apply(temp, 2, statfn))
    attr(temp,'area')        <- attr(x,'area')
    attr(temp,'spacing')     <- attr(x,'spacing')
    if (!is.null(covariates(x))) covariates(temp) <- covariates(x)[subset,,drop=F]
    xl <- range(temp[,1])
    yl <- range(temp[,2])
    attr(temp,'boundingbox') <- expand.grid(x=xl,y=yl)[c(1,2,4,3),]
    class(temp) <- c('mask','data.frame')
    temp
}
############################################################################################

rbind.mask <- function (...) {
# combine 2 or more mask objects
  dropduplicates <- TRUE   ## always
  allargs <- list(...)
  spacing <- attr(allargs[[1]],'spacing')
  area    <- attr(allargs[[1]],'area')
  check <- function (x) {
      if (!is(x,'mask')) 
          stop ('All arguments must be mask objects', call.=F)
      if (attr(x,'spacing') != spacing) 
          stop ('All arguments must have same spacing attribute', call.=F)
      if (attr(x,'area') != area) 
          stop ('All arguments must have same area attribute', call.=F)
      if (!is.null(attr(x,'covariates')))
          warning ('covariates are discarded in rbind.mask', call.=F)
  }
  sapply (allargs, check)

  temp <- rbind.data.frame(...)
  if (dropduplicates) {
      nrw <- nrow(temp)
      temp <- unique(temp)  ## drop duplicate rows
      droppedrows <- nrw - nrow(temp)
      if (droppedrows>0) warning (paste(droppedrows, ' duplicate points dropped from mask'))
  }

  class(temp) <- c('mask', 'data.frame')

    statfn <- function(x) if (is.numeric(x)) c(mean(x), sqrt(var(x))) else c(NA,NA)
    attr(temp,'type')        <- 'rbind'
    attr(temp,'meanSD')      <- as.data.frame (apply(temp, 2, statfn))
    attr(temp,'area')        <- area
    attr(temp,'spacing')     <- spacing
    xl <- range(temp[,1])
    yl <- range(temp[,2])
    attr(temp,'boundingbox') <- expand.grid(x=xl,y=yl)[c(1,2,4,3),]
    class(temp) <- c('mask','data.frame')
    temp
}
############################################################################################

read.mask <- function (file, spacing=NULL, ...)
{
    mask <- read.table (file, ...)
    if (!('x' %in% names(mask)) | !('y' %in% names(mask)))
      names(mask)[1:2] <- c('x','y')  # assume coords in first two columns
    mask <- mask[,c('x','y')]
  
    if (is.null(spacing)) 
    {
      sp      <- as.matrix(dist(as.matrix(mask)))
      spacing <- apply(sp,1,function(x) min(x[x>0]))
      spacing <- mean (spacing, na.rm=T)
    }
  
    area    <- spacing^2 / 10000
  
    xl <- range(mask$x) + c(-spacing/2, +spacing/2)
    yl <- range(mask$y) + c(-spacing/2, +spacing/2)
    
    # get mean and SD of numeric columns
    statfn <- function(x) if (is.numeric(x)) c(mean(x), sqrt(var(x))) else c(NA,NA)
  
    attr(mask,'type')    <- 'user'
    attr(mask,'meanSD')  <- as.data.frame (apply(mask, 2, statfn))
    attr(mask,'area')    <- area
    attr(mask,'spacing') <- spacing 
    attr(mask,'boundingbox') <- expand.grid(x=xl,y=yl)[c(1,2,4,3),]
    attr(mask,'polygon') <- NULL
    class(mask) <- c('mask', 'data.frame')
  
    mask
}
###############################################################################

print.mask <- function(x, ...) {
    if (inherits(x, 'list')) {
        invisible(lapply(x, print, ...))
    }
    else {
        print(data.frame(row.names=1:length(x$x), x=x$x, y=x$y), ...)
    }
}
###############################################################################

plot.mask <- function(x, border=20, add=F, covariate=NULL, 
                     axes=F, dots=T, col='grey', ppoly=T, polycol='red', ...) 
{
    buff <- c(-border,+border)   
    if (!add)
        eqscplot (x$x, x$y, 
        xlim=range(x$x)+buff, ylim=range(x$y)+buff, 
        xlab='', ylab='',
        axes=axes, type='n', ...)

    if (is.null(covariate))
        points (x$x, x$y, col=col, pch=16, cex=0.4)
    else {
        if (length(col)==1) col <- heat.colors(12)   # default set
        if (dots)
            points (x$x, x$y, col = as.numeric(covariates(x)[,covariate]), 
                pch = 16, cex = 0.4)
        else {
            ux <- sort(unique(x$x))
            uy <- sort(unique(x$y))
            temp <- matrix(nc = length(ux), nr = length(uy))
            temp[cbind(match(x$y,uy), match (x$x,ux))] <- as.numeric(covariates(x)[, covariate])
            image(ux, uy, z = temp, col = col, 
                add=add, axes=axes, xlab='', ylab='')
        }
    }

    if ((attr(x,'type')=='polygon') & ppoly) 
        polygon (attr(x,'polygon'), col = polycol, density = 0)
}
###############################################################################

summary.mask <- function(object, ...) {

  if (is.data.frame(object)) {

      if (is.null(object$x) | is.null(object$y)) stop ('Not a valid mask')
      nd <- length(object$x)
      if (length(object$x) != length(object$y)) stop  ('Not a valid mask')
    
      if (!is.null(covariates(object))) {
          sumcovar <- summary(covariates(object), ...)
      } else sumcovar <- NULL
    
      temp <- list (
        detector = attr(object,'detector'), 
        type = attr(object,'type'),  
        nmaskpoints = nrow(object),
        xrange = range(object$x), 
        yrange = range(object$y),
        meanSD = attr(object,'meanSD'),
        spacing = attr(object,'spacing'),
        cellarea = attr(object,'area'),
        boundingbox = attr(object,'boundingbox'),
        covar = sumcovar
      )
      class(temp) <- 'summary.mask'
      temp
  }
  else {
      temp <- lapply(object, summary.mask)
      class(temp) <- c('summary.mask', 'list')
      temp
  }

}
############################################################################################

print.summary.mask <- function (x, ...) {
  if (!('list' %in% class(x))) {
      cat ('Object class     ', 'mask', '\n')
      cat ('Mask type        ', x$type, '\n')
      cat ('Number of points ', x$nmaskpoints, '\n')
      cat ('Spacing m        ', x$spacing, '\n')
      cat ('Cell area ha     ', x$cellarea, '\n')
      cat ('Total area ha    ', x$cellarea * x$nmaskpoints, '\n')
      cat ('x-range m        ', x$xrange, '\n')
      cat ('y-range m        ', x$yrange, '\n')
      cat ('Bounding box     ','\n')
      print (x$boundingbox, ...)  
      cat ('\n')
      if (!is.null(x$covar)) {
          cat ('Summary of covariates', '\n')
          print(x$covar, ...)
      }
  }
  else {
      lapply (x, print.summary.mask)
  }
}
############################################################################################

####################################################
## Class : secr
## spatially explicit capture-recapture model fit
####################################################

secr.fit <- function (capthist, model = list(D~1, g0~1, sigma~1), mask = NULL, 
    buffer = 100, CL = FALSE, detectfn = 0, start = NULL, link = list(), 
    fixed = list(), timecov = NULL, sessioncov = NULL, groups = NULL, 
    dframe = NULL, details = list(), method = 'Newton-Raphson', verify = TRUE, ...) 

{
# Fit spatially explicit capture recapture model 
#
# Arguments:
# 
#  capthist   -  capture history object (includes traps object as an attribute)
#  model      -  formulae for real parameters in terms of effects and covariates
#  mask       -  habitat mask object
#  buffer     -  default buffer width should mask not be provided
#  CL         -  logical switch : conditional likelihood (T) or full likelihood (F)
#  detectfn   -  code for detection function 0 = halfnormal, 1 = hazard, 2 = exponential
#  start      -  start values for maximisation (numeric vector link scale);
#                if NULL then 'autoini' function is used
#  link       -  list of parameter-specific link function names 'log', 'logit', 'identity', 'sin'
#  fixed      -  list of fixed values for named parameters
#  timecov    -  data for time covariates if these are used in 'model'
#  sessioncov -  dataframe of session-level covariates
#  groups     -  vector of names to group fields in attr(capthist,'covariates') dataframe
#  dframe     -  optional data frame of design data for detection model (tricky & untested)
#  details    -  list with several additional settings, mostly of special interest
#  method     -  optimization method (indirectly chooses
#  verify     -  logical switch for pre-check of capthist and mask with verify() 
#  ...        -  other arguments passed to nlm() or optim()

    #################################################
    ## Remember start time and call

    ptm  <- proc.time()
    starttime <- format(Sys.time(), "%H:%M:%S %d %b %Y")  

    cl   <- match.call(expand.dots = TRUE)
    cl <- paste(names(cl)[-1],cl[-1], sep=' = ', collapse=', ' )
    cl <- paste('secr.fit(', cl, ')')

    #################################################
    ## Use input 'details' to override various defaults

    defaultdetails <- list(distribution = 'poisson', scalesigma = FALSE, 
        scaleg0 = FALSE, hessian = 'auto', trace = TRUE, LLonly = FALSE,
        cellprob = FALSE, binomN = 0, cutval = 0, spherical = FALSE, 
        minprob = 1e-50, 
        tx = 'identity')
    details <- replace (defaultdetails, names(details), details)
    if (details$LLonly)  details$trace <- FALSE  

    #################################################
    ## NEW multi-session capthist object 12/2/09
    ## MS - indicator TRUE if multi-session (logical)
    ## sessionlevels - names of sessions (character)

    MS <- inherits(capthist, 'list')
    sessionlevels <- session(capthist)
    if (is.null(sessionlevels)) sessionlevels <- '1'

    if (MS) {
       if (any (sapply(traps(capthist), detector) == 'single')) 
        warning ('multi-catch likelihood used for single-catch traps')
    }
    else {
       if (detector(traps(capthist)) == 'single')
        warning ('multi-catch likelihood used for single-catch traps')
    }   

    #################################################
    ## Optional data check added 2009 09 19
    
    if (verify) {
        memo ('Checking data', details$trace)
        test <- verify(capthist, report = 1)
        if (test$errors) 
            stop ('verify() found errors in capthist argument')

        if (!is.null(mask)) {
            if (MS & !is.data.frame(mask)) {
                ## list of masks
                test <- lapply(mask, verify, report = 1)
                notOK <- any(unlist(test))
            } 
            else notOK <- verify(mask, report = 1)$errors
            if (notOK) 
                stop ('verify() found errors in mask argument')
        }
    }

    #################################################
    ## Ensure valid mask
    ## assume traps(capthist) will extract a list of trap layouts 
    ## if multi-session (MS == TRUE)

    if (is.null(mask)) {
        if (MS) mask <- lapply (traps(capthist), make.mask, buffer = buffer)
        else    mask <- make.mask(traps(capthist), buffer = buffer)         
    }
    else {
      if (MS & is.data.frame(mask)) {
          ## inefficiently replicate mask for each session!
          mask <- lapply(sessionlevels, function(x) mask)
          class (mask) <- c('list', 'mask')
          names(mask) <- sessionlevels
      }      
    }

    nc <- ifelse (MS, sum(sapply(capthist, nrow)), nrow(capthist))
    if (nc < 1) stop (paste(nc,'detection histories'))

    if (MS) 
        q <- attr(capthist[[1]],'q')
    else 
        q <- attr(capthist,'q')
    if (!is.null(q) & CL) stop ('mark-resight incompatible with CL')
 
    #################################################
    ## Use input formula to override defaults

    if ('formula' %in% class(model)) model <- list(model)
    model <- stdform (model)  ## named, no LHS
    defaultmodel <- list(D=~1, g0=~1, sigma=~1, z=~1, w=~1, pID=~1)
    model <- replace (defaultmodel, names(model), model)
    if (! (detectfn %in% c(1))) model$z <- NULL
    if (! (detectfn %in% c(5))) model$w <- NULL
    if (is.null(q)) model$pID <- NULL
    else if (model$pID != ~1) stop ('pID must be constant in this version')

    #################################################
    ## CUSTOMIZE FOR OTHER STYLES OF MODEL - TO DO
    if (CL) model$D <- NULL

    if (CL & !is.null(groups)) {
        groups <- NULL
        warning ('Groups not valid with CL - will ignore') 
    }
    if (CL & ('g' %in% unlist(sapply(model, all.vars)))) stop ('g not a valid effect when CL=TRUE')

    ## Drop any fixed real parameters from model
    model[names(fixed)] <- NULL

    if ((length(model) == 0) & !is.null(fixed)) {
        ## all fixed; assume want only LL
        stop ('all parameters fixed')
    }
    
    vars <-  unlist(lapply(model, all.vars))

    if (details$scalesigma) {
        if (CL) stop ('Cannot use scalesigma with CL')
        if (!is.null(fixed$D)) stop ('Cannot use scalesigma with fixed density')
        if (!(model$D == formula(~1) | 
              model$D == formula(~session))
           ) stop ('Cannot use scalesigma with inhomogenous density or groups')
        if (!is.null(groups)) stop ('Cannot use scalesigma with groups')
    }

    if (details$scaleg0) {
        if (!is.null(groups)) stop ('Cannot use scaleg0 with groups')
    }

    #################################
    # Link functions (model-specific)
    #################################
    defaultlink <- list(D='log', g0='logit', sigma='log', z='log', w='log', pID='sin')
    link <- replace (defaultlink, names(link), link)
    if (details$scaleg0) link$g0 <- 'log'  ## Force log link in this case as no longer 0-1
    if (CL) link$D <- NULL
    if (!(detectfn %in% c(1))) link$z <- NULL 
    if (!(detectfn %in% c(5))) link$w <- NULL 
    if (is.null(q)) link$pID <- NULL

    ##############################################
    # Prepare detection design matrices and lookup
    ##############################################
    memo ('Preparing detection design matrices', details$trace)
    design <- secr.design.MS (capthist, model, timecov, sessioncov, groups, dframe)
    design0 <- secr.design.MS (capthist, model, timecov, sessioncov, groups, dframe, 
        naive = T, bygroup = !CL)   

    ###############################
    # Prepare density design matrix
    ###############################

    D.modelled <- !CL & is.null(fixed$D)
    if (!D.modelled) {
       D.designmatrix <- matrix(nr=0, nc=0)
       grps <- NULL
       attr(D.designmatrix, 'dimD') <- NA
    }
    else {
        memo ('Preparing density design matrix', details$trace)
        grps  <- group.levels(capthist,groups)
        temp <- D.designdata( mask, model$D, grps, sessionlevels, sessioncov)
        D.designmatrix <- model.matrix(model$D, temp) 
        attr(D.designmatrix, 'dimD') <- attr(temp, 'dimD')
    }

    #############################
    # Parameter mapping (general)
    #############################

    np <- sapply(design$designMatrices, ncol)
    if (D.modelled) np <-  c(D = ncol(D.designmatrix), np)
    NP <- sum(np)
    parindx <- split(1:NP, rep(1:length(np), np)) 
    names(parindx) <- names(np)
    if (!D.modelled) parindx$D <- NULL

    #############################
    # Single evaluation option
    #############################

    .localstuff$iter <- 0
    if (details$LLonly) {
      if (is.null(start)) stop('Must provide transformed parameter values in start')
      if (!is.null(q)) stop ('not for mark-resight')
      
      LL <- - secr.loglikfn (beta = start,
                       link       = link,
                       fixed      = fixed,
                       parindx    = parindx,
                       capthist   = capthist,
                       mask       = mask,
                       CL         = CL,
                       detectfn   = detectfn,
                       designD    = D.designmatrix,
                       design     = design,
                       design0    = design0,
                       groups     = groups,
                       details    = details,
                       logmult    = logmultinom(capthist, group.factor(capthist, groups))    
                       )
      return(c(logLik=LL))
    }

    ## 2009 10 17 tentative insertion
    if (details$cellprob) {
      if (is.null(start)) stop('Must provide transformed parameter values in start')
      if (!is.null(q)) stop ('not for mark-resight')    
      pi.n <- secr.cellprob (beta = start, link = link, fixed = fixed,
          parindx = parindx, capthist = capthist, mask = mask, CL = CL, 
          detectfn = detectfn, designD = D.designmatrix, design = design, 
          design0 = design0, groups = groups, details = details)
      return(pi.n)
    }
    ## 2009 10 17 tentative insertion ends

    ###############################
    # Start values (model-specific)
    ###############################
    ## 'start' is vector of beta values (i.e. transformed)
    if (!is.null(start)) stopifnot (length(start) == NP)
    else {
        memo('Finding initial parameter values...', details$trace)

        ## autoini uses default buffer dbar * 4
        if (MS) {
            if (ncol(capthist[[1]])==1) stop('autoini not implemented for 1 occasion; specify start')
            start3 <- autoini (capthist[[1]], mask[[1]]) ## Use session 1 - can be risky
        }
        else {
            if (ncol(capthist)==1) stop('autoini not implemented for 1 occasion; specify start')
            start3 <- autoini (capthist, mask)
        }
        memo(paste('Initial values ', paste(round(unlist(start3),5),collapse=',')), details$trace)

        if (any(is.na(start3))) {
            warning ('secr.fit failed because initial values not found (data sparse?)')
            return (list(call=cl, fit=NULL))
        }

        ## next two stmts must be this order (g0 then sigma)
        if (details$scaleg0) start3$g0 <- start3$g0 * start3$sigma^2
        if (details$scalesigma) start3$sigma <- start3$sigma * start3$D^0.5 

        start <- rep(0, NP)
        if (D.modelled) start[1] <- transform(start3$D, link$D)      # /ngrp when figure out where to calculate this
        if (is.null(fixed$g0)) start[parindx$g0[1]]    <- transform(start3$g0, link$g0)
        if (is.null(fixed$sigma)) start[parindx$sigma[1]] <- transform(start3$sigma, link$sigma)

        if (detectfn==1 & is.null(fixed$z)) start[parindx$z[1]] <- transform(5, link$z)
        if (detectfn==5 & is.null(fixed$w)) start[parindx$w[1]] <- transform(10, link$w)
        if (!is.null(q) & is.null(fixed$pID)) start[parindx$pID[1]] <- transform(0.7, link$pID)
    }

    ##########################
    # Fixed beta parameters
    ##########################
    fb <- details$fixedbeta
    if (!is.null(fb)) {
        if (!(length(fb)== NP))
            stop ('invalid fixed beta - require NP-vector')
        if (sum(is.na(fb))==0)
            stop ('cannot fix all beta parameters')
        start <- start[is.na(fb)]  ## drop unwanted betas; remember later to adjust parameter count
    }

    ##########################
    # Variable names (general)
    ##########################
    betanames <- unlist(sapply(design$designMatrices, colnames))
    names(betanames) <- NULL
    realnames <- names(model)
    if (D.modelled) betanames <- c(paste('D', colnames(D.designmatrix), sep='.'), betanames)
    betanames <- sub('..(Intercept))','',betanames)
    ## allow for fixed beta parameters 2009 10 19
    if (!is.null(details$fixedbeta))
        betanames <- betanames[is.na(details$fixedbeta)]

    #################################
    # Variable names (model-specific)
    #################################

    if (details$scaleg0)     {
        betanames <- sub('g0','g0*', betanames)
        realnames <- sub('g0','g0*', realnames)
    }
    if (details$scalesigma)  {
        betanames <- sub('sigma','sigma*', betanames)
        realnames <- sub('sigma','sigma*', realnames)
    }

    betaw <- max(max(nchar(betanames)),8)   # for 'trace' formatting

    #####################
    # Maximize likelihood
    #####################   
    savedlogmultinomial <- logmultinom(capthist, group.factor(capthist, groups))    

    memo('Maximizing likelihood...', details$trace)
    if (details$trace) cat('Eval     Loglik', formatC(betanames, format='f', width=betaw), '\n')

    ## mark - resight option

    if (is.null(q))
        loglikefn <- secr.loglikfn
    else
        loglikefn <- MRsecr.loglikfn

    if (tolower(method) %in% c('newton-raphson', 'nr')) {
        this.fit <- nlm (p         = start,
                        f          = loglikefn,
                        link       = link,
                        fixed      = fixed,
                        parindx    = parindx,
                        capthist   = capthist,
                        mask       = mask,
                        CL         = CL,
                        detectfn   = detectfn,
                        designD    = D.designmatrix, 
                        design     = design,
                        design0    = design0,
                        groups     = groups,
                        details    = details,
                        logmult    = savedlogmultinomial,
                        betaw      = betaw,   # for trace format
                        hessian    = tolower(details$hessian)=='auto',
                        ...)  
        this.fit$par <- this.fit$estimate     # copy for uniformity
        this.fit$value <- this.fit$minimum    # copy for uniformity
    }
    else {
        this.fit <- optim (par     = start,
                        fn         = loglikefn,
                        link       = link,
                        fixed      = fixed,
                        parindx    = parindx,
                        capthist   = capthist,
                        mask       = mask,
                        CL         = CL,
                        detectfn   = detectfn,
                        designD    = D.designmatrix, 
                        design     = design,
                        design0    = design0,
                        groups     = groups,
                        details    = details,
                        logmult    = savedlogmultinomial,
                        betaw      = betaw,   # for trace format
                        hessian    = tolower(details$hessian)=='auto',
                        method     = method,
                        ...)   # default method = 'BFGS', control=list(parscale=c(1,0.1,5))
    }

    ############################
    # Variance-covariance matrix
    ############################

    if (tolower(details$hessian)=='fdhess') {
        memo ('Computing Hessian with fdHess in nlme', details$trace)
        loglikfn <- function (beta) {
           -secr.loglikfn(
                        beta     = beta,
                        link       = link,
                        fixed      = fixed,
                        parindx    = parindx,
                        capthist   = capthist,
                        mask       = mask,
                        CL         = CL,
                        detectfn   = detectfn,
                        designD    = D.designmatrix, 
                        design     = design,
                        design0    = design0,
                        groups     = groups,
                        details    = details,
                        logmult    = savedlogmultinomial,
                        betaw      = betaw,   # for trace format
                 )
        }
        grad.Hess <- fdHess(this.fit$par, fun = loglikfn, .relStep = 0.001, minAbsPar=0.1)
        this.fit$hessian <- -grad.Hess$Hessian
    }

    this.fit$method <- method  ## remember what method we used...

    if (!is.null(this.fit$hessian)) {
        covar <- try(solve(this.fit$hessian))
        if (inherits(covar, "try-error")) {
            warning ('could not invert Hessian to compute variance-covariance matrix')
            covar <- matrix(rep(NA,NP*NP), nc=NP)  # failed      
        }  
        dimnames(covar) <- list(betanames, betanames)
    }
    else covar <- NULL
  
    ## predicted D across mask
    ## added 2009 09 04
    if (CL) D <- NULL
    else {
        D <- getD (D.designmatrix, this.fit$par, mask, parindx, link, fixed, MS, length(grps), length(sessionlevels))
        dimnames (D) <- list(1:nrow(D), grps, sessionlevels)
        ## N <- t(apply(D, 2:3, sum, drop = FALSE))
        ## dimnames (N) <- list(sessionlevels, grps)
    }

    desc <- packageDescription("secr")  ## for version number

    temp <- list (call = cl, 
                  capthist = capthist, 
                  mask = mask, 
                  detectfn = detectfn, 
                  CL = CL, 
                  timecov = timecov,
                  sessioncov = sessioncov,
                  groups = groups, 
                  dframe = dframe,
                  design = design,      ## added 2009 09 05
                  design0 = design0,    ## added 2009 06 25

                  start = start,        ## added 2009 09 09
                  link = link, 
                  fixed = fixed, 
                  parindx = parindx, 
                  model = model, 
                  details = details,

                  vars = vars, 
                  betanames = betanames, 
                  realnames = realnames,

                  fit = this.fit, 
                  beta.vcv = covar, 
                  D = D,                   ## added 2009 09 04
                  version = desc$Version,  ## added 2009 09 21
                  starttime = starttime,   ## added 2009 09 21
                  proctime = (proc.time() - ptm)[1]

             )

    attr (temp, 'class') <- 'secr'
    memo(paste('Completed in ', round(temp$proctime,2), ' seconds at ', 
        format(Sys.time(), "%H:%M:%S %d %b %Y"),
        sep=''), details$trace)
    temp

}
############################################################################################

trim.secr <- function (object, drop = c('mask','design','design0','D'), keep = NULL) {
    trim.default(object, drop = drop, keep = keep)
}
############################################################################################

predict.secr <- function (object, newdata = NULL, se.fit = TRUE, alpha = 0.05, 
    savenew = FALSE, ...) {

    if (is.null(newdata)) newdata <- secr.make.newdata (object)

    ## allow for fixed beta parameters 2009 10 19
    fb <- object$details$fixedbeta
    if (!is.null(fb)) {
        nbeta <- length(fb)
        beta.vcv <- matrix(0, nr=nbeta, nc=nbeta)
        beta.vcv[is.na(fb[row(beta.vcv)]) & is.na(fb[col(beta.vcv)])] <- object$beta.vcv
        fb[is.na(fb)] <- object$fit$par
        beta <- fb    ## complete
    }
    else {
        beta <- object$fit$par
        beta.vcv <- object$beta.vcv 
    }

    getfield <- function (x) secr.lpredictor (newdata = newdata, model = object$model[[x]], 
        indx = object$parindx[[x]], beta = beta, beta.vcv = beta.vcv)
    predict <- sapply (names(object$model), getfield, simplify=FALSE)

    z <- abs(qnorm(1-alpha/2))   ## beware confusion with hazard z!
    if (se.fit)  out <- list(nrow(newdata))
    else { 
        out <- newdata
        ## add columns for real parameter estimates
        for (varname in object$realnames) 
            out[,varname] <- rep(NA,nrow(out))
    }

    for (new in 1:nrow(newdata)) {
        lpred   <- sapply (predict, function(x) x[new,'estimate'])

        Xlpred  <- Xuntransform(lpred, object$link, object$realnames)
        if (se.fit) {
            selpred <- sapply (predict,function(x) x[new,'se'])
            temp <- data.frame (
              row.names = object$realnames,
              link = unlist(object$link[object$realnames]),
              estimate = Xlpred,
              SE.estimate = se.Xuntransform (lpred, selpred, object$link, object$realnames),
              lcl = Xuntransform(lpred-z*selpred, object$link, object$realnames),
              ucl = Xuntransform(lpred+z*selpred, object$link, object$realnames)
              )
            # truncate density at zero
            if (!object$CL) temp['D', -1][temp['D',-1]<0] <- 0 
    
            if (nrow(newdata)==1) out <- temp
            else {
                out[[new]] <- temp
                names(out)[new] <- paste (
                        paste(names(newdata),'=', unlist(lapply(newdata[new,],as.character)), 
                        sep=' ',collapse=', '),
                    sep=',')
            }
        }
        else  # no SE; terse format
        {
            out[new, (ncol(newdata)+1) : ncol(out)] <- Xlpred
        } 
    }
    if (savenew) attr(out, 'newdata') <- newdata
    out
}

############################################################################################

coef.secr <- function (object, alpha=0.05, ...) {
    beta   <- object$fit$par
    if (!is.null(object$beta.vcv))
        sebeta <- suppressWarnings(sqrt(diag(object$beta.vcv)))
    else sebeta <- rep(NA, length(beta))
    z <- abs(qnorm(1-alpha/2))
    temp <- data.frame(
        row.names = object$betanames,
        beta    = beta, 
        SE.beta = sebeta,
        lcl = beta - z*sebeta,
        ucl = beta + z*sebeta
        )
    attr(temp, 'alpha') <- alpha
    temp
}
############################################################################################

model.string <- function (model) {

## parentheses removed 2009 08 18
##    temp <- paste (names(model), as.character(model), collapse=') ', sep='(')
##    temp <- paste(temp,')',sep='')

    temp <- paste (names(model), as.character(model), collapse=' ', sep='')

    temp
}
fixed.string <- function (fixed) {
    if (is.null(fixed) | length(fixed)==0) 'none'
    else paste (names(fixed), as.character(fixed), collapse=', ', sep=' = ')    
}
############################################################################################

print.secr <- function (x, newdata = NULL, alpha = 0.05, deriv = FALSE, ...) {

    cat ('\n')
    cat (x$call, '\n')
    if (!is.null(x$version)) {
        cat ('secr ', x$version, ', ', x$starttime, '\n', sep='')
    }
    else {   ## for backward compatibility
        cat (x$fitted,'\n')
    }

    cat ('\n')

    ###################
    ## Data description

    MS <- inherits(x$capthist, 'list')
    if (MS) {
        n     <- sapply(x$capthist, nrow)        # number caught
        nocc  <- sapply(x$capthist, ncol)        # number occasions
        ncapt <- sapply(x$capthist, function (xx) sum(abs(xx)>0))
        ndet  <- sapply(traps(x$capthist), nrow) # number traps
        temp  <- as.data.frame(rbind(nocc, ncapt, n, ndet))
        names(temp) <- names(x$capthist)
        rownames(temp) <- c('Occasions','Captures','Animals','Detectors')
        q <- sapply(x$capthist, function(y) attr(y,'q'))
        print(temp)
    }
    else {

        n  <- nrow(x$capthist)     # number caught
        ncapt <- sum(abs(x$capthist)>0)
        q <- attr(x$capthist, 'q')

        if ('g' %in% x$vars) {
            Groups  <- table(group.factor(x$capthist, x$groups))
            temp <- paste (names(Groups), Groups, collapse=', ', sep='=')
            temp <- paste('(',temp,')', sep='')        
        }
        else temp <- ''

        cat ('N animals       : ', n, temp, '\n')
        cat ('N captures      : ', ncapt, '\n')
        if (!is.null(q)) {
            cat ('N marking  occn : ', q, '\n')
            cat ('N sighting occn : ', ncol(x$capthist)-q, '\n')
        }
        else 
            cat ('N occasions     : ', ncol(x$capthist), '\n')
        cat ('N detectors     : ', nrow(traps(x$capthist)), '\n')

    }   # end of single-session

    cat ('\nDetector type   : ',  detector(traps(x$capthist)), '\n')
    if (!MS)
    cat ('Mask area       : ', attr(x$mask,'area') * nrow(x$mask), 'ha \n\n')

    ####################
    ## Model description

    Npar <- max(unlist(x$parindx))
    ## allow for fixed beta parameters 2009 10 19
    if (!is.null(x$details$fixedbeta))
        Npar <- Npar - sum(!is.na(x$details$fixedbeta))

    cat ('Model           : ', model.string(x$model), '\n')
    cat ('Fixed (real)    : ', fixed.string(x$fixed), '\n')
    cat ('Detection fn    : ', switch(x$detectfn+1, 'halfnormal', 'hazard rate','exponential',
                               'compound halfnormal','uniform','w exponential',,,,,
                               'signal strength','binary signal strength'), '\n')
    if (!x$CL)
    cat ('Distribution    : ', x$details$distribution, '\n')

    cat ('N parameters    : ', Npar, '\n')
    cat ('Log likelihood  : ', -x$fit$value, '\n')
    cat ('AIC             : ', 2*(x$fit$value + Npar), '\n')
    cat ('AICc            : ', 2*(x$fit$value + Npar) + 2 * Npar * (Npar+1) / (sum(n) - Npar - 1), '\n')

    cat ('\n')
    cat ('Beta parameters (coefficients)', '\n')
    print(coef(x), ...)

    if (!is.null(x$fit$hessian)) {
      cat ('\n')
      cat ('Variance-covariance matrix of beta parameters', '\n')
      print (x$beta.vcv, ...)
    }

    # scale newdata covariates... NOT FINISHED 10 05 08
    meanSD <- attr(x$mask,'meanSD')
    if (!is.null(newdata)) { 
         for (i in 1:length(newdata)) { 
           ind <- match (names(newdata[i]),names(meanSD))
           if (ind>0 & !is.na(meanSD[1,ind]))
             newdata[[i]] <- (newdata[[i]] - meanSD[1,ind]) / meanSD[2,ind]
         }
     }

    cat ('\n')
    cat ('Fitted (real) parameters evaluated at base levels of covariates', '\n')

    if (!is.null(x$realpar))
        print( x$realpar )
    else {
        temp <- predict (x, newdata, alpha) 
        nd <- length(temp)
        if (is.data.frame(temp)) print(temp, ...)
        else for (new in 1:nd) {
                cat('\n', names(temp)[new],'\n')
                print(temp[[new]], ...)
            }
    }

    #################################
    # Derived parameters (CL)
    #################################
    if (x$CL & deriv) {

        cat ('\n')
        cat ('Derived parameters (CL only)', '\n')

        temp <- derived(x, alpha=alpha, se.esa = TRUE)
        nd <- length(temp)
        if (is.data.frame(temp)) print(temp, ...)
        else for (new in 1:nd) {
                cat('\n',names(temp)[new],'\n')
                print(temp[[new]], ...)
            }

    }
    cat ('\n')
}
############################################################################################

oneline.secr <- function (secr) {

    if (is.list(secr$capthist)) {
        n <- sum(sapply (secr$capthist, nrow))
        ncapt <- sum(sapply (secr$capthist, function (x) sum(abs(x>0))))
    }
    else  {
        n  <- nrow(secr$capthist)     # number caught
        ncapt <- sum( abs( secr$capthist)>0)
    }

    Npar <- max(unlist(secr$parindx))
    ## allow for fixed beta parameters 2009 10 19
    if (!is.null(secr$details$fixedbeta))
        Npar <- Npar - sum(!is.na(secr$details$fixedbeta))

    c (
       model  = model.string(secr$model),
       detectfn = c('halfnormal','hazard rate','exponential','compound halfnormal',
           'uniform','w exponential')[secr$detectfn+1],
       npar   = Npar,
       logLik = -secr$fit$value,
       AIC    = round(2*(secr$fit$value + Npar), 3), 
       AICc   = round(2*(secr$fit$value + Npar) + 2 * Npar * (Npar+1) / (n - Npar - 1), 3),
       fitted = secr$fitted
    )
}
############################################################################################

AIC.secr <- function (object, ..., sort = TRUE, k = 2, dmax = 10) {

    if (k != 2) stop ('AIC.secr defined only for k = 2')

    allargs <- list(...)
    modelnames <- (c ( as.character(match.call(expand.dots=FALSE)$object),
          as.character(match.call(expand.dots=FALSE)$...) ))
    if (any(sapply(allargs,class) != 'secr')) stop ('All arguments must be secr objects', call.=F)
    allargs <- c(list(object), allargs)


#    if (!all.equal(sapply (allargs, function(x) detector(traps(x$capthist)))))
#    stop('Models not compatible')
#    if (!all.equal(sapply (allargs, function(x) x$CL)))
#    stop('Models not compatible')

    output <- data.frame(t(sapply(allargs, oneline.secr)), stringsAsFactors=F)
    for (i in 3:6)
    output[,i] <- as.numeric(output[,i])
    output$dAICc <- output$AICc - min(output$AICc)

    OK <- abs(output$dAICc) < abs(dmax)
    sumdAICc <- sum(exp(-output$dAICc[OK]/2))
    output$AICwt <- ifelse ( OK, round(exp(-output$dAICc/2) / sumdAICc,4), 0)

    row.names(output) <- modelnames
    if (sort) output <- output [order(output$AICc),]

    if (nrow(output)==1) { output$dAICc <- NULL; output$AICwt <- NULL}
    output
}
############################################################################################

vcov.secr <- function (object, realnames = NULL, newdata = NULL, byrow = FALSE, ...) {
## return either the beta-parameter variance-covariance matrix
## or vcv each real parameters between points given by newdata (byrow = TRUE)
## or vcv for real parameters at points given by newdata (byrow = TRUE)

    if (is.null(dimnames(object$beta.vcv))) 
        dimnames(object$beta.vcv) <- list(object$betanames, object$betanames)

    if (is.null(realnames))  
        ## average beta parameters
        return( object$beta.vcv )
    else {
        ## average real parameters
        ## vcv among multiple rows

        if (byrow) {
            ## need delta-method variance of reals given object$beta.vcv & newdata
            if (is.null(newdata)) 
                newdata <- secr.make.newdata (object)
            nreal <- length(realnames)
            nbeta <- length(object$fit$par)

            rowi <- function (newdatai) {
                reali <- function (beta, rn) {
                    ## real from all beta pars eval at newdata[i,] 
                    par.rn <- object$parindx[[rn]]
                    mat <- model.matrix(object$model[[rn]], data=newdatai)
                    lp <- mat %*% matrix(beta[par.rn], nc = 1)
                    untransform (lp, object$link[[rn]])
                }  
                grad <- matrix(nr = nreal, nc = nbeta) 
                dimnames(grad) <- list(realnames, object$betanames)
                for (rn in realnames)
                    grad[rn,] <- fdHess (pars = object$fit$par, fun = reali, rn = rn)$gradient
                vcv <- grad %*% object$beta.vcv %*% t(grad)
                vcv
            }

            vcvlist <- list(nrow(newdata)) 
            for (i in 1:nrow(newdata)) vcvlist[[i]] <- rowi(newdata[i,])
            if (length(vcvlist) == 1) vcvlist <- vcvlist[[1]]
            return(vcvlist)
        }
        else {
            newdata <- as.data.frame(newdata)
            rownames <- apply(newdata, 1, function(x) paste(names(newdata), '=', x, sep='', collapse=',')) 
            vcvlist <- list()
            for (rn in realnames) {
                par.rn <- object$parindx[[rn]]
                mat <- model.matrix(object$model[[rn]], data=newdata)
                lp <- mat %*% matrix(object$fit$par[par.rn], nc = 1)
                real <- untransform (lp, object$link[[rn]])
                real <- as.vector(real)
                ## from Jeff Laake's 'compute.real' in RMark...
                deriv.real <- switch(object$link[[rn]],
                    logit = mat * real * (1-real),
                    log = mat * real,
                    identity = mat,
                    sin = mat * cos(asin(2*real-1))/2) 
                vcvlist[[rn]] <- deriv.real %*% object$beta.vcv[par.rn, par.rn] %*% t(deriv.real)          
                dimnames(vcvlist[[rn]]) <- list(rownames, rownames)
            }
            names (vcvlist) <- realnames
            return (vcvlist)
        }
        ## DIFFERENT VARIANCE TO secr.lpredictor for sigma because there use se.Xuntransfom
    }
}
############################################################################################

