############################################################################################
## package 'secr'
## sim.capthist.R
## simulate capture histories
## last changed 2009 06 19, 2009 09 09 'seed' added
## sim.resight 2009 10 08
############################################################################################

expand <- function (x, n, q = 0) {
    if (is.null(x)) NA
    else {
        y <- numeric(n)
        if ((length(x)==2) & (q>0)) 
            y[] <- rep(x, c(q,n-q))
        else 
            y[] <- x 
        y
    }
}

sim.capthist <- function (
    traps, 
    popn = list(D = 5, buffer = 100, Ndist = 'poisson'), 
    detectfn = 0, 
    detectpar = list(g0 = 0.2, sigma = 25, z = 1), 
    noccasions = 5, 
    renumber = TRUE,
    seed = NULL
    )

#
# Simulate detection of known population
#
# To do 05 06 08
# take account of usage attribute in traps

{

    ##################
    ## set random seed
    ## copied from simulate.lm
    if (!exists(".Random.seed", envir = .GlobalEnv, inherits = FALSE)) 
        runif(1)
    if (is.null(seed)) 
        RNGstate <- get(".Random.seed", envir = .GlobalEnv)
    else {
        R.seed <- get(".Random.seed", envir = .GlobalEnv)
        set.seed(seed)
        RNGstate <- structure(seed, kind = as.list(RNGkind()))
        on.exit(assign(".Random.seed", R.seed, envir = .GlobalEnv))
    }
    ##################

    if (is.null(detector(traps))) 
        stop ("Not a valid traps: detector type needed")

    if (!is.null(usage(traps))) {
        if (ncol(usage(traps)) != noccasions) {
            noccasions <- ncol(usage(traps))
            warning ("Ignoring 'noccasions' - does not match usage attribute of 'traps'", call.=F) 
        }
    }

    if (detector(traps) == 'count') {
        binomN <- attr(traps, 'binomN')   # 0 for Poisson, >0 for binomial
        if (is.null(binomN)) stop ('Count detectors must have attribute binomN', call.=F)
        if (binomN < 0) {
            binomN <- 0
            warning ("Invalid number of binomial trials; using Poisson instead")
        }
    }

    validatepar <- function (x, xrange) {
        xname <- deparse(substitute(x))
        if (is.null(x)) stop (paste('No value for',xname), call.=F)
        if (any(is.na(x))) stop (paste('NA is not a valid value for',xname), call.=F)
        if (any(x < xrange[1])) warning (paste('Value for',xname,'is less than minimum',xrange[1]), call.=F)
        if (any(x > xrange[2])) warning (paste('Value for',xname,'is greater than maximum',xrange[2]), call.=F)  
    }

    # Detection function parameters

    if (detectfn %in% c(0,1,2,3)) {
        g0 <- expand(detectpar$g0, noccasions, 0)
        sigma <- expand(detectpar$sigma, noccasions, 0)
        if (detectfn == 1) z <- expand(detectpar$z, noccasions, 0)
        else z <- rep(1, noccasions)
        validatepar(g0, c(0,1))
        validatepar(sigma, c(1e-10,Inf))
        validatepar(z, c(1e-10,Inf))
    }

    # Acoustic detection function parameters

    if (detectfn %in% c(10,11)) {

        tx <- detectpar$tx
        if (is.null(tx)) tx <- 'log'
        spherical <- detectpar$spherical
        if (is.null(spherical)) spherical <- FALSE

        beta0 <- expand(detectpar$beta0, noccasions, 0)
        beta1 <- expand(detectpar$beta1, noccasions, 0)
        sdS   <- expand(detectpar$sdS, noccasions, 0)
        cut   <- detectpar$cut
        validatepar(beta0, c(-Inf,Inf))
        validatepar(beta1, c(-Inf,Inf))
        validatepar(sdS, c(0,Inf))
        validatepar(cut, c(-Inf,Inf))      
    }
  

  if (!is(popn,'popn')) # generate if not provided
  {
      defaultpopn <- list(D = 5, buffer = 100, Ndist = 'poisson')
      popn <- replace (defaultpopn, names(popn), popn)
      popn <- sim.popn (popn$D, core=traps, buffer=popn$buffer, covariates=NULL, Ndist = popn$Ndist)
  }
  N <- nrow(popn)
  animals <- unlist(popn) 

  k <- nrow(traps)

  if (detector(traps) == 'proximity') {
      if (detectfn==11) {  # binary signal strength
          # stop ('Signal strength detector not available in secr 1.2')
          temp <- .C("trappingsignal", PACKAGE = 'secr',
            as.double(beta0),
            as.double(beta1), 
            as.double(sdS), 
            as.double(cut),
            as.integer(noccasions),    
            as.integer(k), 
            as.integer(N), 
            as.double(animals), 
            as.double(unlist(traps)), 
            as.integer(detectfn), 
            as.integer(spherical),
            n = integer(1),
            caught = integer(N),
            value = double(N*noccasions*k),
            resultcode = integer(1)
          )     
      }
      else {

          # cat ('g0         ', g0, '\n')
          # cat ('sigma ', sigma, '\n')
          # cat ('detectfn ', detectfn, '\n')
          # cat ('z ', z, '\n')
          # cat ('noccasions ', noccasions, '\n')
          # cat ('k ', k, '\n')
          # cat ('N ', N, '\n')

         temp <- .C("trappingproximity", PACKAGE = 'secr',
            as.double(g0),
            as.double(sigma), 
            as.double(z), 
            as.integer(noccasions),    
            as.integer(k), 
            as.integer(N), 
            as.double(animals), 
            as.double(unlist(traps)), 
            as.integer(detectfn), 
            n = integer(1),
            caught = integer(N),
            value = integer(N*noccasions*k),
            resultcode = integer(1)
          )
    }

    if (temp$resultcode!=0) 
        stop (paste('simulated proximity detection failed, code ', temp$resultcode))
    w <- array(dim=c(temp$n, noccasions, k), dimnames = list(NULL,1:noccasions,NULL))
    if (temp$n>0)  w[,,] <- round(temp$value[1:(temp$n*noccasions*k)])
  } 

  else
  if (detector(traps) == 'signal') {
      temp <- .C("trappingsignal", PACKAGE = 'secr',
        as.double(beta0),
        as.double(beta1), 
        as.double(sdS), 
        as.double(cut),
        as.integer(noccasions),    
        as.integer(k), 
        as.integer(N), 
        as.double(animals), 
        as.double(unlist(traps)), 
        as.integer(detectfn), 
        as.integer(spherical),
        n = integer(1),
        caught = integer(N),
        value = double(N*noccasions*k),
        resultcode = integer(1)
    )
    if (temp$resultcode != 0) stop ('Call to trappingsignal failed')
    w <- array(NA, dim=c(temp$n, noccasions, k), dimnames = list(NULL,1:noccasions,NULL))

    if (temp$n>0)  w[,,] <- temp$value[1:(temp$n * noccasions * k)]

    if (detectfn == 10) {
        # stop ('Signal strength detector not available in secr 1.1')
        # large negative values indicate non-detection
        # regardless of transformation  
        detected <- (w > -1e10)
    
        w[detected] <- switch( tx, 
            log = exp(w[detected]),
            logit = invlogit( w[detected] ),
            identity = w[detected]
        )
        w[!detected] <- 0
    } 
  }
  else
  if (detector(traps) == 'count') {
      temp <- .C("trappingcount", PACKAGE = 'secr',
        as.double(g0),
        as.double(sigma), 
        as.double(z), 
        as.integer(noccasions),    
        as.integer(k), 
        as.integer(N), 
        as.double(animals), 
        as.double(unlist(traps)), 
        as.integer(detectfn), 
        as.integer(binomN),
        n = integer(1),
        caught = integer(N),
        value = integer(N*noccasions*k),
        resultcode = integer(1)
      )
      if (temp$resultcode != 0) stop ('Call to trappingcount failed')
      w <- array(dim=c(temp$n, noccasions, k), dimnames = list(NULL,1:noccasions,NULL))
      if (temp$n>0)  w[,,] <- temp$value[1:(temp$n*noccasions*k)]
  }
  else
  if (detector(traps) == 'areasearch') {
      if (is.null(attr(traps,'spacing'))) stop ('Cell spacing required')
      cellsize <- attr(traps,'spacing')^2 * 0.0001
      temp <- .C("trappingarea", PACKAGE = 'secr',
        as.double(g0),
        as.double(sigma), 
        as.double(z), 
        as.integer(noccasions),    
        as.integer(k), 
        as.integer(N), 
        as.double(animals), 
        as.double(unlist(traps)), 
        as.double(cellsize),
        as.integer(detectfn), 
        n = integer(1),
        caught = integer(N),
        value = integer(N*noccasions*k),
        resultcode = integer(1)
      )
      if (temp$resultcode != 0) stop ('Call to trappingarea failed')
      w <- array(dim=c(temp$n, noccasions, k), dimnames = list(NULL,1:noccasions,NULL))
      if (temp$n>0)  w[,,] <- temp$value[1:(temp$n*noccasions*k)]
  }
  else 
  if (detector(traps)=='multi') {
      temp <- .C("trappingmulti", PACKAGE = 'secr',
          as.double(g0),
          as.double(sigma), 
          as.double(z), 
          as.integer(noccasions),    
          as.integer(k), 
          as.integer(N), 
          as.double(animals), 
          as.double(unlist(traps)), 
          as.integer(detectfn), 
          n = integer(1),
          caught = integer(N),
          value=integer(N*noccasions),
          resultcode = integer(1)
      )
      if (temp$resultcode != 0) stop ('Call to trappingmulti failed')
      w <- matrix(nr = temp$n, nc = noccasions, dimnames = list(NULL,1:noccasions))
      if (temp$n > 0) w[,] <- temp$value[1:(temp$n*noccasions)]
  }
  else if (detector(traps)=='single') {
      temp <- .C("trappingsingle", PACKAGE = 'secr',
          as.double(g0),
          as.double(sigma), 
          as.double(z), 
          as.integer(noccasions),    
          as.integer(k), 
          as.integer(N), 
          as.double(animals), 
          as.double(unlist(traps)), 
          as.integer(detectfn), 
          n = integer(1),
          caught = integer(N),
          value=integer(N*noccasions),
          resultcode = integer(1)
        )
        if (temp$resultcode != 0) stop ('Call to trappingsingle failed')
        w <- matrix(nr = temp$n, nc=noccasions, dimnames = list(NULL,1:noccasions))
        if (temp$n > 0) w[,] <- temp$value[1:(temp$n*noccasions)]    
      }
  else stop ('Unrecognised detector type')

  if (!is.null(covariates(popn))) {
      covariates(w) <- covariates(popn)[as.logical(temp$caught),, drop=F]
  }
  
  class(w)               <- 'capthist'    # NOT data.frame
  traps(w)               <- traps
  attr(w, 'cut')         <- cut           # new 2008 08 09 
  attr(w,'seed')         <- RNGstate   ## save random seed

  # dummy session values for now
  session(w) <- '1'

  if (renumber & (temp$n>0)) rownames(w) <- 1:temp$n
  else rownames(w)          <- (1:N)[as.logical(temp$caught)]

  w
}
############################################################################################

sim.resight <- function (traps, ..., q = 1, pID = 1) {

    defaultpar <- list(noccasion = 5)
    dots <- list(...)
    dots <- replace (defaultpar, names(dots), dots)
    dots$detectpar$g0 <- expand (dots$detectpar$g0, dots$noccasion, q)
    dots$detectpar$sigma <- expand (dots$detectpar$sigma, dots$noccasion, q)
    dots$detectpar$z <- expand (dots$detectpar$z, dots$noccasion, q)

    capthist <- do.call('sim.capthist', c(list(traps = traps), dots))

    ## transform simulated capthist object into resighting data
    S <- ncol(capthist)
    K <- nrow(traps(capthist))

    if (S <= q) stop('no sighting intervals')
    if (!(detector(traps(capthist)) %in% c('proximity'))) 
        stop ('only for proximity detectors')

    ## sighting only unmarked animals
    marked <- apply(capthist[,1:q,,drop = FALSE], 1, sum) > 0
    ## always get warning about no detections on first occasion
    suppressWarnings(R <- subset(capthist, subset=!marked, dropnull=F))
    tempM <- subset(capthist, subset=marked, dropnull=F)
    nM <- nrow(tempM)
    ID <- abind(tempM[,1:q, , drop=F], array(runif(nrow(tempM)*(S-q)*K) < pID, dim=c(nM,S-q,K)), along=2)
    ## marking and sighting, marked animals
    M <- ifelse(ID, tempM,0)    ## ID
    U <- ifelse(!ID, tempM,0)   ## notID
    dimnames(M)[[2]] <- 1:S
    countfn <- function(x) {
        x <- x * col(x)   ## x 0/1
        tabulate(x[x>0], nbin=K)
    }
    Tm <- apply(U, 2, countfn)  ## notID, marked
    Tu <- apply(R, 2, countfn)  ## not marked
    row.names(Tu) <- row.names(traps(capthist))    
    dimnames(Tm) <- dimnames(Tu)

    class(M) <- 'capthist'
    session(M) <- session(capthist)
    traps(M) <- traps(capthist)

    attr(M, 'Tu') <- Tu
    attr(M, 'Tm') <- Tm
    attr(M, 'q') <- q

    M
}
############################################################################################

