############################################################################################
## package 'secr'
## verify.R
## 2009 09 18, 2009 09 19, 2009 09 20 2009 10 02
############################################################################################

verify <- function (object, report, ...) UseMethod("verify")

verify.default <- function (object, report, ...) {
  cat ('no verify method for objects of class', class(object), '\n')
}
############################################################################################

verify.traps <- function (object, report = 2, ...) {

## Check internal consistency of 'traps' object
## 
## -- Number of rows in dataframe of detector covariates differs expected
## -- Number of detectors in usage matrix differs from expected
## -- Occasions with no used detectors

    if (!inherits(object, 'traps')) {
         stop ('object must be of class traps')
    }

    if (inherits(object, 'list')) {
        temp <- lapply (object, verify, report = min(report,1))
        anyerrors <- any(sapply(temp, function(x) x$errors))
        if ((report == 2) & !anyerrors)
            cat('No errors found :-)\n')
        invisible(list(errors = anyerrors, bysession = temp))
    }
    else {
 
        dim3 <- detector(object) %in% c('proximity')
        single <- detector(object) %in% c('single')
        usagedetectorsOK <- TRUE
        usagenonzeroOK <- TRUE

        if (!is.null(covariates(object))) 
            if ((ncol(covariates(object)) == 0 ) | 
                (nrow(covariates(object)) == 0 )) covariates(object) <- NULL

        ## 1
        trapNAOK <- !any(is.na(object))


        ## 2
        trapcovariatesOK <- ifelse (is.null(covariates(object)), 
            TRUE, nrow(covariates(object)) == nrow(object))

        ## 'usage' of traps
        if (!is.null(usage(object))) {
            ## 3
            if (dim3) 
                usagedetectorsOK <- nrow(usage(object)) == dim(object)[3]
            else 
                usagedetectorsOK <- nrow(usage(object)) == nrow(object)            

            ## 4
            usagecount <- apply(usage(object),2,sum)
            usagenonzeroOK <- !any(usagecount == 0)
        }
        else usagecount <- rep(NA, ncol(object))

        errors <- !all(c(trapcovariatesOK, usagedetectorsOK, usagenonzeroOK))

        if (report > 0) {
            if (errors) {
                if (!trapNAOK) {
                    cat ('Missing detector coordinates not allowed\n')
                }
                if (!trapcovariatesOK) {
                    cat ('Wrong number of rows in dataframe of detector covariates\n')
                    cat ('traps(capthist) :', nrow(traps(object)), 'detectors\n')
                    cat ('covariates(traps(capthist)) :', nrow(covariates(traps(object))), 'detectors\n')
                }
                if (!usagedetectorsOK) {
                    cat ('Conflicting number of detectors in usage matrix\n')
                    cat ('traps(capthist) :', nrow(traps(object)), 'detectors\n')
                    cat ('usage(traps(capthist)) :', nrow(usage(traps(object))), 'detectors\n')
                }
                if (!usagenonzeroOK) {
                    cat ("Occasions when no detectors 'used'\n")
                    cat ((1:length(usagecount))[usagecount==0], '\n')
                }
            }
        }

        if ((report == 2) & !errors) cat('No errors found :-)\n')

        out <- list(errors = errors, 
            trapNAOK = trapNAOK,
            trapcovariatesOK = trapcovariatesOK, 
            usagedetectorsOK = usagedetectorsOK,
            usagenonzeroOK = usagenonzeroOK,
            usagecount = usagecount
        )
        invisible(out)

    }
}
############################################################################################

verify.capthist <- function (object, report = 2, ...) {
 
## Check internal consistency of 'capthist' object
## 
## -- 'traps' component present
## -- verify(traps)
## -- No live releases
## -- Live detection(s) after reported dead
## -- More than one capture in single-catch trap(s)

## -- Number of rows in 'traps' object not compatible with reported detections
## -- Number of rows in dataframe of individual covariates differs from capthist
## -- Number of occasions in usage matrix differs from capthist
## -- Detections at unused detectors


    if (!inherits(object, 'capthist')) stop ('object must be of class capthist')

    if (inherits(object, 'list')) {
        temp <- lapply (object, verify, report = min(report, 1))
        anyerrors <- any(sapply(temp, function(x) x$errors))
        if ((report == 2) & !anyerrors)
            cat('No errors found :-)\n')
        invisible(list(errors = anyerrors, bysession = temp))
    }
    else {
 
        ## preliminaries
        dim3 <- detector(traps(object)) %in% c('proximity')  ## later may include other detector types
        prox <- detector(traps(object)) %in% c('proximity')
        single <- detector(traps(object)) %in% c('single')
        usageOK <- TRUE
        usageoccasionsOK <- TRUE
        usagedetectorsOK <- TRUE
        usagenonzeroOK <- TRUE
        detectorconflcts <- NULL
        duplicateOK <- TRUE
        proximityOK <- TRUE
 
        if (!is.null(covariates(object))) 
            if ((ncol(covariates(object)) == 0 ) | 
                (nrow(covariates(object)) == 0 )) 
                covariates(object) <- NULL

        ## 1
        trapspresentOK <- !is.null(traps(object))

        ## standalone check of detectors
        if (trapspresentOK)
            trapcheck <- verify(traps(object), report = 0)  ## delay reporting
        else
            trapcheck <- list(errors=TRUE)

        ## 2
        trapsOK <- !trapcheck$errors 

        ## 3
        detectionsOK <- sum(object[object>0]) > 0

        ## 4
        NAOK <- !any(is.na(object))

        ## 5
        fn <- function(x) {
            if (dim3) x <- apply(x,1,min)
            (min(x)<0) & (tail(x[x!=0],1)>0)
        }
        undead <- apply(object, 1, fn)
        deadOK <- !any(undead)
        if (!deadOK) {
            if (dim3)
                reincarnated <- object[undead,,, drop=F]
            else
                reincarnated <- object[undead,, drop=F]
        }

        ## 6
        if (single) {
            fn <- function (x) duplicated(abs(x)[x!=0])
            multiple <- apply(object, 2, fn)
            duplicateOK <- !any(unlist(multiple))
        }

        ## 7
        if (prox) {
            multiples <- sum(abs(object)>1)
            proximityOK <- multiples == 0
        }

        ## 8
        detectornumberOK <- ifelse (dim3,
              dim(object)[3] == nrow(traps(object)),
              max(abs(object)) <= nrow(traps(object)))  

        ## 9
        covariatesOK <- ifelse(is.null(covariates(object)), 
            TRUE, 
            nrow(covariates(object)) == nrow(object))

        ## is 'usage' of traps consistent with reported detections?
        if (!is.null(usage(traps(object)))) {
            conflcts <- 0

            ## 10 
            usageoccasionsOK <- ncol(usage(traps(object))) == ncol(object)
    
            notused <- !usage(traps(object))  ## traps x occasions

            if (dim3) {
                 
                if (usagedetectorsOK & usageoccasionsOK) {
                    tempobj <- aperm(object, c(2,3,1))   ## occasion, traps, animal sKn
                    tempuse <- array(t(usage(traps(object))), dim=dim(tempobj))  ## replicated to fill...
                    conflcts <- (abs(tempobj)>0) & (tempuse==0)
                    occasion <- rep(row(tempobj[,,1]), dim(tempobj)[3])
                    detector <- rep(col(tempobj[,,1]), dim(tempobj)[3])
                    ID <- rep(rownames(object), rep(prod(dim(tempobj)[1:2]), nrow(object)))
                    detectorconflcts <- as.data.frame(cbind(ID,detector,occasion)[conflcts,])
                }
            }
            else {
                if (usagedetectorsOK & usageoccasionsOK) { 
                    OK <- as.numeric(object)>0
                    occasion <- as.numeric(col(object))[OK]
                    ID <- row.names(object)[as.numeric(row(object))[OK]]
                    detector <- as.numeric(object)[OK]
                    conflcts <- notused[cbind(detector, occasion)] > 0            
                    detectorconflcts <- as.data.frame(cbind(ID,detector,occasion)[conflcts,])
                }
            }

            ## 11
            usageOK <- sum(conflcts)==0   

        }

        errors <- !all(c(trapspresentOK, trapsOK, detectionsOK, NAOK, deadOK, duplicateOK, proximityOK, detectornumberOK, covariatesOK, 
            usageoccasionsOK, usageOK))

        if (report > 0) {
            if (errors) {
                cat ('Session', session(object), '\n')

                if (!trapspresentOK) {
                    cat ('No valid detectors\n')
                }
                if (!trapsOK) {
                    cat ('Errors in traps\n')               
                    if (!trapcheck$trapNAOK) {
                        cat ('Missing detector coordinates not allowed\n')
                    }
                    if (!trapcheck$trapcovariatesOK) {
                        cat ('Wrong number of rows in dataframe of detector covariates\n')
                        cat ('traps(capthist) :', nrow(traps(object)), 'detectors\n')
                        cat ('covariates(traps(capthist)) :', nrow(covariates(traps(object))), 'detectors\n')
                    }
                    if (!trapcheck$usagedetectorsOK) {
                        cat ('Conflicting number of detectors in usage matrix\n')
                        cat ('traps(capthist) :', nrow(traps(object)), 'detectors\n')
                        cat ('usage(traps(capthist)) :', nrow(usage(traps(object))), 'detectors\n')
                    }
                    if (!trapcheck$usagenonzeroOK) {
                        cat ("Occasions when no detectors 'used'\n")
                        cat ((1:length(trapcheck$usagecount))[trapcheck$usagecount==0], '\n')
                    }
                }

                if (!detectionsOK) {
                    cat ('No live releases\n')
                }

                if (!NAOK) {
                    cat ('Missing values not allowed in capthist\n')
                }

                if (!deadOK) {
                    cat ('Recorded alive after dead\n')
                    print(reincarnated) 
                }

                if (!duplicateOK) {
                    cat ('More than one capture in single-catch trap(s)\n')                   
                }

                if (!proximityOK) {
                    cat ('More than one detection per detector per occasion at proximity detector(s)\n')                   
                }

                if (!detectornumberOK) {
                    cat ('traps object incompatible with reported detections\n')
                    cat ('traps(capthist) :', nrow(traps(object)), 'detectors\n')
                    if (dim3)
                        cat ('capthist :', dim(object)[3], 'detectors\n')
                    else 
                        cat ('capthist :', max(abs(object)), 'max(detector)\n')
                }  

                if (!covariatesOK) {
                    cat ('Wrong number of rows in dataframe of individual covariates\n')
                    cat ('capthist :', nrow(object), 'individuals\n')
                    cat ('covariates(capthist) :', nrow(covariates(object)), 'individuals\n')
                }
                if (!usageoccasionsOK) {
                    cat ('Conflicting number of occasions in usage matrix\n')
                    cat ('capthist :', ncol(object), 'occasions\n')
                    cat ('usage(traps(capthist)) :', ncol(usage(traps(object))), 'occasions\n')    
                }
                if (!usageOK) {
                    cat ("Detections at 'unused' detectors\n")
                    print(detectorconflcts)
                }
            }

            if ((report == 2) & !errors) cat('No errors found :-)\n')

        }

        out <- list(errors = errors, trapcheck = trapcheck)
        if (!is.null(detectorconflcts)) out$detections.at.unused.detectors <- detectorconflcts
        invisible(out)
    }
}
############################################################################################

verify.mask <- function (object, report = 2, ...) {
 
## Check internal consistency of 'mask' object
## 
## valid x and y coordinates
## nrow(covariates) = nrow(object)
## ...also look at attributes? 

    if (!inherits(object, 'mask')) stop ('object must be of class mask')

    if (inherits(object, 'list')) {
        temp <- lapply (object, verify, report = min(report, 1))
        anyerrors <- any(sapply(temp, function(x) x$errors))
        if ((report == 2) & !anyerrors)
            cat('No errors found :-)\n')
        invisible(list(errors = anyerrors, bysession = temp))
    }
    else {
 
        ## 1
        xyOK <- !(is.null(object$x) | is.null(object$y) | any(is.na(object)))
        xyOK <- xyOK & is.numeric(unlist(object))

        ## 2
        
        if (!is.null(covariates(object)))
            covariatesOK <- ifelse (nrow(covariates(object))>0,
            nrow(object) == nrow(covariates(object)), TRUE)
        else
            covariatesOK <- TRUE

        errors <- !all(c(xyOK, covariatesOK))

        if (report > 0) {
            if (errors) {
                ## cat ('Session', session(object), '\n')

                if (!xyOK) {
                    cat ('Invalid x or y coordinates in mask\n')
                }
 
                if (!covariatesOK) {
                    cat ('Number of rows in covariates(mask) differs from expected\n')
                }
            }

            if ((report == 2) & !errors) cat('No errors found :-)\n')
        }

        out <- list(errors = errors)
        invisible(out)
    }
}
############################################################################################

