/* 
   External procedures for secr package
   Murray Efford 2008 07 10 - 2009 09 26
   2009 09 10 single-catch trap simulation integrated with simsecr!
   2009 09 24 remove local math functions & rely on R
   2009 09 26 convert all comments to C (not C++ double slash)
   2009 09 26 un-mix declarations and code in integralprw1 & secrloglik
   2009 09 30 remaining code incompatible with ISO C: variable-length arrays  
   2009 09 30 amendments by Ray Brownrigg to compile on Unix
              - removed all mention of EXPORT
              - changed random() to Random  

   This version does not include code developed in July-September 2008 for 
   area detectors, signal strength etc.
   See MGE file d:\\single sample\\paper\\signal.c, especially secrloglik0 
   (also d:\\density secr 1.2\\oldcode)

   can compile with gcc 4.2.1-sjlj
   gcc -Ic:/progra~1/R/R-2.9.2/include -c secr.c -Wall -pedantic -std=gnu99
*/

#include "secr.h"
#include <math.h>    
#include <stdlib.h>
#include <stdio.h> 
#include <R.h>       /* 2009 09 08 random numbers */
#include <Rmath.h>   /* 2009 09 24 R math functions e.g. dbinom, dpois */

double huge = 1e10;
double fuzz = 1e-200;
double minimumexp = -100;

struct trap_animal {                   
        int     trap;
        int     animal;
        double  time;
};

FILE *out;           

/*==============================================================================*/

double Random () { 

/*
   this call to R requires preceding 
   GetRNGstate();
   and following
   PutRNGstate();
*/
    return( unif_rand() );     /* 2009 09 08 */
}
/*==============================================================================*/

double ldpois (int k, double l)
/* log Poisson density */
{
    return (dpois (k, l, 1));
}
/*==============================================================================*/

double ldbinom (int k, double p, double N)
/* log binomial density */
{
    return(dbinom ( k, N, p, 1));
}
/*==============================================================================*/

double expmin (double x)
{
  if (x < minimumexp)
      return(0);
  else 
      return(exp(x));  
}
/*==============================================================================*/

double d2 (
    int k, 
    int m, 
    double A1[], 
    double A2[], 
    int A1rows, 
    int A2rows)
/*
   return squared distance between two points given by row k in A1
   and row m in A2, where A1 and A2 have respectively A1rows and A2rows
*/
{
    return(
        (A1[k] - A2[m]) * (A1[k] - A2[m]) +
        (A1[k + A1rows] - A2[m + A2rows]) * (A1[k + A1rows] - A2[m + A2rows])
    );
}

/*==============================================================================*/

double mufn (
    int k, 
    int m, 
    double b0, 
    double b1,
    int spherical,
    double A1[], 
    double A2[], 
    int A1rows, 
    int A2rows)
/*
   Return predicted signal strength at m for source at point k, 
   given strength at source of b0 dB and attenuation of b1 dB/m.
   Spherical spreading is included if spherical > 0
   Coordinates of points are in A1 and A2 which have respectively 
   A1rows and A2rows
*/
{
    double d2val;
    d2val = d2(k,m, A1, A2, A1rows, A2rows);  
     if ((spherical>0) & (d2val>1)) {            
        return (b0 - 10 * log ( d2val ) / 2.302585 + b1 * (sqrt(d2val)-1));  /* checked 2008 08 08 */
    }
    else 
        return (b0 + b1 * sqrt(d2val));
}
/*==============================================================================*/

double ghn (
    int k, 
    int m, 
    int c, 
    double gsbval[], 
    int cc, 
    double traps[], 
    double mask[], 
    int kk, 
    int mm, 
    double cut, 
    int spherical)
{
    return (gsbval[c] * exp(-d2(k,m,traps,mask,kk,mm) / 2 / 
        gsbval[cc + c] / gsbval[cc + c]));  
}

double ghz (
    int k, 
    int m, 
    int c, 
    double gsbval[], 
    int cc, 
    double traps[], 
    double mask[], 
    int kk, 
    int mm, 
    double cut, 
    int spherical)
{
    return (gsbval[c] * (1 - exp(- pow(sqrt(d2(k,m,traps,mask,kk,mm)) / gsbval[cc + c], 
        - gsbval[cc * 2 + c])))); 
}

double ghe (
    int k, 
    int m, 
    int c, 
    double gsbval[], 
    int cc, 
    double traps[], 
    double mask[], 
    int kk, 
    int mm, 
    double cut, 
    int spherical)
{
    return (gsbval[c] * exp(-sqrt(d2(k,m,traps,mask,kk,mm)) / gsbval[cc + c]));  
}

double ghf (
    int k, 
    int m, 
    int c, 
    double gsbval[], 
    int cc, 
    double traps[], 
    double mask[], 
    int kk, 
    int mm, 
    double cut, 
    int spherical)

/* new 'flat-topped exponential' 2009 09 01 */
{
    double d, w, g0, sigma;
    d = sqrt(d2(k,m,traps,mask,kk,mm));
    g0 = gsbval[c];
    sigma = gsbval[cc + c];
    w = gsbval[cc * 2 + c]; 
    if (d<w) return (g0);
    else return (g0 * exp(-(d-w) / sigma));  
}
 /*===============================================================*/

double gphi (
    int k, 
    int m, 
    int c, 
    double gsbval[], 
    int cc, 
    double traps[], 
    double mask[], 
    int kk, 
    int mm, 
    double cut, 
    int spherical)
{
    double mu, gam, sdS;
    mu = mufn (k, m, gsbval[c], gsbval[cc + c], spherical,
         traps, mask, kk, mm);
    sdS = gsbval[cc * 2 + c];
    gam = (cut - mu) / sdS; 
    return (pnorm(gam,0,1,0,0));    /* upper */
}
/*===============================================================*/

double pndot (int m, int n, int ncol, int gsb0[], double gk0[], int ss, int kk, int cc0)
/*
    probability animal at point m on mask is caught
    n may indicate group (full likelihood; ncol= number of groups) or 
    individual (conditional likelihood; ncol= number of individuals)
    aligned with secrloglik 2009 06 25
*/
{
    int k,s,c;
    double pp=1;
    for (k=0; k< kk; k++)
        for (s=0; s<ss; s++)
        { 
            c = gsb0[ncol * (ss * k + s) + n] - 1; 
            if (c >= 0)     /* drops unset traps */
            pp *= 1 - gk0[cc0 * (kk * m + k) + c];
        }
    return (1 - pp);
}
/*===============================================================*/

void integralprw1 (
    double *gsb0val,   /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) [naive animal] */
    int    *nc,        
    int    *ss,        /* number of occasions */
    int    *kk,        /* number of traps */
    int    *mm,        /* number of points on mask */
    double *traps,     /* x,y locations of traps (first x, then y) */
    double *mask,      /* x,y points on mask (first x, then y) */
    int    *cc0,       /* number of g0/sigma/b combinations [naive animal] */
    int    *gsb0,      /* lookup which g0/sigma/b combination to use for given n, S, K [naive animal] */
    int    *ncol,      /* number of columns in gsb0; added 2009 06 25 */
    double *area,      /* area associated with each mask point (ha) */
    int    *fn,        /* code 0 = halfnormal, 1 = hazard, 2 = exponential, 10 = signal strength, 11 = binary signal strength */
    int    *binomN,    /* number of trials for 'count' detector modelled with binomial */
    double *cut,       /* transformed signal strength threshold for detection */
    int    *spherical,
    double *a,         /* return value integral of pr(w0) */
    int    *resultcode /* 0 for successful completion */
)
  
{
    int n,k,m,c;
    double (*gfn) (
        int k, 
        int m, 
        int c, 
        double gsbval[], 
        int cc, 
        double traps[], 
        double mask[], 
        int kk, 
        int mm, 
        double cut, 
        int spherical);
    double asum = 0;
    double *gk0;

    /* MAINLINE */

    *resultcode = 1;   /* generic failure */

    /* Allocate space for array of naive detection probability */
    gk0 = malloc(*cc0 * *kk * *mm * sizeof (double));

    if (gk0 == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }

    /* set functions appropriate to options */
    if (*fn == 0) gfn = ghn;  
    else if (*fn == 1) gfn = ghz;
    else if (*fn == 2) gfn = ghe;
    else if (*fn == 5) gfn = ghf;
    else if ((*fn == 10) | (*fn == 11)) gfn = gphi;
    else return; /* detection function not recognised */

    for (c=0; c<*cc0; c++)
      for (k=0; k<*kk; k++)
        for (m=0; m<*mm; m++) 
          gk0[*cc0 * (*kk * m + k) + c] = gfn(k, m, c, gsb0val,
              *cc0, traps, mask, *kk, *mm, *cut, *spherical);

    for (n=0; n<*nc; n++)    /* CH numbered 0 <= n < *nc */
    {   
        if ((*ncol > 1) | (n == 0))   /* no need to repeat if constant */
        {
            if ((n+1) > *ncol)   /* groups not implemented */
            { *resultcode = 3;  return; }

            asum = 0;
            for (m=0; m<*mm; m++) 
            {
                asum += pndot (m, n, *ncol, gsb0, gk0, *ss, *kk, *cc0);
            }
        }
        a[n] = *area * asum;
    }

    free (gk0);

    *resultcode = 0;   /* successful completion */
}

/*==============================================================================*/

double hxy (
    int c, 
    int m,
    double gk[],
    int cc,
    int kk)
/*
    Sum hazard over traps for animal at m
    for parameter combination c
*/
{
    int k;
    double pp;
    double sumhz = 0;
    for (k=0; k<kk; k++)
    {
        /* probability caught in trap k 0 <= k < kk */
        pp = gk[cc * (kk * m + k) + c];
        if (pp > (1-fuzz))    /* pp close to 1.0 - approx limit */
            pp = huge;        /* g0 very large (effectively infinite hazard) */
        else {
            if (pp <= 0) pp = 0;
            else pp = -log(1-pp);
        }
        sumhz += pp;         /* accumulate hazard over traps */
    }
    return (sumhz);
}
/*========================================================*/

double hxys (
    int c, 
    int m, 
    int s, 
    int gsb[], 
    double gk[],
    int cc,
    int nc, 
    int ss, 
    int kk)

/*
    Sum hazard over traps for animal at m
    for parameter combination c
    with traps in use on occasion s
*/
{
    int k;
    double pp;
    double sumhz = 0;
    for (k=0; k<kk; k++)
    {
        /* check animal 0 to see if this trap set on occ s */
        /* skip if not set */
        if (gsb[nc * (ss * k + s)] > 0) {
            /* probability caught in trap k 0 <= k < kk */
            pp = gk[cc * (kk * m + k) + c];
            if (pp > (1-fuzz))    /* pp close to 1.0 - approx limit */
                pp = huge;        /* g0 very large (effectively infinite hazard) */
            else {
                if (pp <= 0) pp = 0;
                else pp = -log(1-pp);
            }
            sumhz += pp;         /* accumulate hazard over traps */
        }
    }
    return (sumhz);
}
/*========================================================*/

double prwiprox (
    int m, 
    int n,
    int w[],
    int gsb[],
    double gk[],
    double hxytemps[],
    int cc, 
    int nc,
    int kk,
    int ss,
    int mm,
    double minp)
/*
   Probability of capture history n (0 <= n < *nc)
   given that animal's range centre is at m
   PROXIMITY DETECTOR
*/
{
    int s;   /* index of occasion  0 <= s < *ss */
    int k;   /* index of trap      0 <= k < *kk */
    int c, indx;
    int state;
    int dead = 0;
    double result = 1.0;      
    for (s=0; s<ss; s++)
    { 
        for (k=0; k<kk; k++)
        {
            state = w[nc * (ss * k + s) + n];
            if (state<0) {dead=1; state=-state;}
            c = gsb[nc * (ss * k + s) + n] - 1;  
            if (c >= 0) {       /* skip if this trap not set */
                indx  = cc * (kk * m + k) + c;                                
                if (state>0) result *= gk[indx];
                else result *=  1.0 - gk[indx];
                if (result < minp) {result = minp; break;}  /* truncate */
            }
        }
        if (result < minp) {result = minp; break;}      /* truncate */
        if (dead) break;
    }
    return (result);
}

/*=============================================================*/
double prwimulti (
    int m, 
    int n,
    int w[],
    int gsb[],
    double gk[],
    double hxytemps[],
    int cc, 
    int nc,
    int kk,
    int ss,
    int mm,
    double minp)
/*
    Probability of capture history n (0 <= n < *nc)
    given that animal's range centre is at m
    MULTI-CATCH DETECTOR
*/
{
    int s;   /* index of occasion  0 <= s < *ss */
    int k;   /* index of trap      0 <= k < *kk */
    int c;
    int indx;
    int dead = 0;
    double htemp;
    double pks;
    double result = 1.0;
  
    for (s=0; s<ss; s++)
    {
        k = w[nc * s + n];
        if (k < 0) {dead=1; k=-k;}  /*  1 <= k <= kk */
        if (k > 0) {
            c = gsb[nc * (ss * (k-1) + s) + n] - 1;  
            indx = cc * (kk * m + (k-1)) + c;
            htemp = hxytemps[cc * (mm * s + m) + c];
            if (htemp < 1e-200) { result = 0; break; }
            pks = -log (1 - gk[indx]);
            pks *= (1-expmin(-htemp)) / htemp;   
        }
        else  {
            c = gsb[nc * s + n] - 1;  /* use combination for k=0 - as good as any */
            htemp = hxytemps[cc * (mm * s + m) + c]; 
            if (htemp < 1e-200) { result = 0; break; }
            pks = expmin(-htemp);  /* Not captured */
        }
        result *= pks;
        if (dead) break;
    }
    return (result);
}  
/*=============================================================*/

void secrloglik (
    int    *like,        /* likelihood 0 full, 1 conditional */
    int    *detect,      /* detector 0 multi, 1 proximity */
    int    *distrib,     /* distribution 0 Poisson, 1 binomial */
    int    *w,           /* capture histories (1:nc, 1:s, 1:k) */
    int    *grp,         /* group number for 0<=n<*nc   [full likelihood only] */
    int    *nc,          /* number of capture histories */
    int    *ss,          /* number of occasions */
    int    *kk,          /* number of traps */
    int    *mm,          /* number of points on mask */
    int    *gg,          /* number of groups */
    double *traps,       /* x,y locations of traps (first x, then y) */
    double *mask,        /* x,y points on mask (first x, then y) */
    double *Dmask,       /* density at each point on mask, possibly x group */
    double *gsbval,      /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) */
    double *gsb0val,     /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) [naive animal] */
    int    *cc,          /* number of g0/sigma/b combinations  */
    int    *cc0,         /* number of g0/sigma/b combinations for naive animals */
    int    *gsb,         /* lookup which g0/sigma/b combination to use for given n, S, K */
    int    *gsb0,        /* lookup which g0/sigma/b combination to use for given g, S, K [naive animal] */
    double *area,        /* area associated with each mask point (ha) */
    int    *fn,          /* code 0 = halfnormal, 1 = hazard, 2 = exponential */
    int    *binomN,      /* number of trials for 'count' detector modelled with binomial */
    double *cut,         /* transformed signal strength threshold for detection */
    int    *spherical,
    double *minprob,     /* minimum value of P(detection history) */
    double *a,           /* a(theta) */
    double *value,       /* return value integral of pr(w0) */
    int    *resultcode   /* 0 if OK */
)

{
    int    i,n,g,k,m,c,s;
    int    ng[*gg];       /* number per group */
    int    nSK;
    int    notset = 0;
    int    indiv = 0;     /* indicator for whether detection varies between individuals (CL only) */
    double asum = 0;
    double temp;
    double sumD[*gg];     /* 2009 08 21  for binomial */
    double sumDp[*gg];

    double *gk;
    double *gk0;
    double *hxytemps;
    double (*gfn) (
        int k, 
        int m, 
        int c, 
        double gsbval[], 
        int cc, 
        double traps[], 
        double mask[], 
        int kk, 
        int mm, 
        double cut, 
        int spherical);

    double (*prwfn) (
        int m, 
        int n,
        int w[],
        int gsb[],
        double gk[],
        double hxytemps[],
        int cc, 
        int nc,
        int kk,
        int ss,
        int mm,
        double minp);

    /*===============================================================*/

    int debug=0;

    /* MAINLINE */

    if (debug) {
        out = fopen( "d:\\debug1.txt", "w" );
        fprintf(out, "Start\n");
        fflush(out);
    }

    *resultcode = 1;  /* generic failure code */
                      /* reset to 0 at end */

    gk = malloc(*cc * *kk * *mm * sizeof (double));
    if (gk == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }
    
    gk0 = malloc(*cc * *kk * *mm * sizeof (double));
    if (gk0 == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }

    /*  allocate enough space to allow for the case where some 
        traps are not set */
    hxytemps = malloc (*cc * *mm * *ss * sizeof(double));
    if (hxytemps == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }


    /* set functions appropriate to options */
    if (*detect == 0) prwfn = prwimulti; else prwfn = prwiprox;

    if (*fn == 0) gfn = ghn;  
    else if (*fn == 1) gfn = ghz;
    else if (*fn == 2) gfn = ghe;
    else if (*fn == 5) gfn = ghf;
    else return;

    for (k=0; k<*kk; k++)
      for (m=0; m<*mm; m++) {
          for (c=0; c<*cc0; c++)
              gk0[*cc0 * (*kk * m + k) + c] = gfn(k, m, c, gsb0val, *cc0, 
                  traps, mask, *kk, *mm, *cut, *spherical);
          for (c=0; c<*cc; c++)
              gk[*cc * (*kk * m + k) + c] = gfn(k, m, c, gsbval, *cc, 
                  traps, mask, *kk, *mm, *cut, *spherical);
      }

 
    if (*detect == 0)                           /* only for multicatch traps */
    {
        /* check if any traps not set */
        /* and decide whether to use full or collapsed hxy */
        nSK = *nc * *ss * *kk;
        for (i=0; i<nSK; i++) if (gsb[i]==0) notset++;

        if (notset==0) {
          for (c=0; c<*cc; c++)
            for (m=0; m<*mm; m++) 
            {
              hxytemps[*cc * m + c] = hxy(c,m, gk, *cc, *kk);
              for (s=1; s<*ss; s++)
                  hxytemps[*cc * (*mm * s + m) + c] = hxytemps[*cc * m + c];
            }
        }
        else {    /* some traps not set: occasion-specific total hazard */
          for (s=0; s<*ss; s++)  
            for (c=0; c<*cc; c++)
              for (m=0; m<*mm; m++) 
                hxytemps[*cc * (*mm * s + m) + c] = hxys(c,m,s, gsb, gk, *cc, *nc, *ss, *kk);
        }
    }

    if (*like==1)    /* Conditional likelihood */
    {
        /*
           check if we need to consider variation among individuals
           i.e. check if detection parameters constant within a CH
        */
        indiv = 0;
        for (s=0; s<*ss; s++)  
          for (k=0; k<*kk; k++)
          {   
              i = gsb0[*nc * (*ss * k + s)];
              for (n=1; n<*nc; n++)
                  if (i != gsb0[*nc * (*ss * k + s) + n]) {
                      indiv = 1; break;
                  }
          }

        if (indiv == 0) {
            /* save time by doing this once, rather than inside n loop */
            asum = 0;
            for (m=0; m<*mm; m++) 
            {
                asum += pndot (m, 0, *nc, gsb0, gk0, *ss, *kk, *cc0);   /* all individuals the same */
            }
        }
        /* else asum calculated for each individual in loop below */

        *value = 0;
        /* Loop over individuals... */
        for (n=0; n<*nc; n++)    /* CH numbered 0 <= n < *nc */
        {   
            temp = 0;
            if (indiv > 0) asum = 0;
            for (m=0; m<*mm; m++) 
            {
                temp += prwfn (m, n, w, gsb, gk, hxytemps, *cc, *nc, *kk, *ss, *mm, *minprob);
                if (indiv > 0) asum += pndot (m, n, *nc, gsb0, gk0, *ss, *kk, *cc0);
            }
            a[n] = *area * asum;     /* a[n] checked against Density 6/6/08 */
            *value += log(temp/asum);

            if (debug) {
                fprintf(out, "n %10d temp %20.15f asum %20.15f logLik %10.5f \n", n, temp, asum, log(temp/asum));
                fflush(out);
            }

        }
        /* ...end of loop over individuals */
    }

    else  /* *like==0  Full likelihood */
    {
        for (g=0; g<*gg; g++) {
            ng[g] = 0;  /* zero for later */
            sumD[g] = 0;
            sumDp[g] = 0;
            for (m=0; m<*mm; m++)  {
                sumD[g] += Dmask[*mm * g + m];
                sumDp[g] += Dmask[*mm * g + m] * pndot (m, g, *gg, gsb0, gk0, *ss, *kk, *cc0);
            }
        }

        *value = 0;
        for (n=0; n<*nc; n++) /* CH are numbered 0 <= n < *nc in C code */
        {   
            g = grp[n]-1;
            ng[g]++; 
            temp = 0;
            for (m=0; m<*mm; m++) 
            {
                temp += prwfn (m, n, w, gsb, gk, hxytemps, *cc, *nc, *kk, *ss, *mm, *minprob) * 
                    Dmask[*mm * g + m];    
            }
            *value += log(temp);
        }

        for (g=0; g<*gg; g++) {
            *value -= ng[g] * log(sumDp[g]);
            /* Poisson */
            if (*distrib==0) *value += ldpois(ng[g], sumDp[g] * *area); 
            /* binomial */
            if (*distrib==1) *value += ldbinom (ng[g], sumDp[g] / sumD[g], sumD[g] * *area);
        }
    }

    /* free dynamic memory allocated with malloc */
    free (gk);
    free (gk0);
    free (hxytemps);

    if (debug) {
        fprintf(out, "\n");
        fclose(out);
    }

    *resultcode = 0;   /* successful termination secrloglik */
}
/*==============================================================================*/

double MRprwiprox (
    int m, 
    int n,
    int w[],
    int gsb[],
    double gk[],
    double pID,
    double hxytemps[],
    int cc, 
    int nc,
    int kk,
    int ss,
    int qq,
    int mm,
    double minp)
/*
   Probability of capture history n (0 <= n < *nc)
   given that animal's range centre is at m
   PROXIMITY DETECTOR

   Note no differentiation between marking phase (occasions 1:qq) and
   later: both here taken to involve binary proximity detection.

*/
{
    int s;   /* index of occasion  0 <= s < *ss */
    int k;   /* index of trap      0 <= k < *kk */
    int c, indx;
    int state;
    int dead = 0;
    double result = 1.0;      
    for (s=0; s<ss; s++)
    { 
        for (k=0; k<kk; k++)
        {
            state = w[nc * (ss * k + s) + n];
            if (state<0) {dead=1; state=-state;}
            c = gsb[nc * (ss * k + s) + n] - 1;  
            if (c >= 0) {       /* skip if this trap not set */
                indx  = cc * (kk * m + k) + c;  
                              
                if (state>0) {
                    result *= gk[indx];
                    if (s>qq) result *= pID;  /* risk of not ID */
                }

                else result *=  1.0 - gk[indx];
                if (result < minp) {result = minp; break;}  /* truncate */
            }
        }
        if (result < minp) {result = minp; break;}      /* truncate */
        if (dead) break;
    }
    return (result);
}

/*=============================================================*/

double MRprwimulti (
    int m, 
    int n,
    int w[],
    int gsb[],
    double gk[],
    double pID,
    double hxytemps[],
    int cc, 
    int nc,
    int kk,
    int ss,
    int qq, 
    int mm,
    double minp)
/*
    Probability of capture history n (0 <= n < *nc)
    given that animal's range centre is at m
    MULTI-CATCH DETECTOR for s = 1..qq
    SIGHTING DETECTOR for s = qq+1..ss
*/
{
    int s;   /* index of occasion  0 <= s < *ss */
    int k;   /* index of trap      0 <= k < *kk */
    int c;
    int indx;
    int dead = 0;
    double htemp;
    double pks;
    double result = 1.0;
    int state;
  
    /* marking phase,  occasions 1:qq */
    for (s=0; s<qq; s++)
    {
        /* get k - custom code to adapt 3-D input */       
        state = 0;
        for (k=0; k<kk; k++) {
             state = w[nc * (ss * k + s) + n];
             if (state != 0) break;   /* first nonzero */
        }
        if (abs(state)>0) k = sign(state) * (k+1);
        else k = 0; 
        /* end of get k */

        if (k < 0) {dead=1; k=-k;}             /*  1 <= k <= kk */
        if (k > 0) {
            c = gsb[nc * (ss * (k-1) + s) + n] - 1;  
            indx = cc * (kk * m + (k-1)) + c;
            htemp = hxytemps[cc * (mm * s + m) + c];
            if (htemp < 1e-200) { result = 0; break; }
            pks = -log (1 - gk[indx]);
            pks *= (1-expmin(-htemp)) / htemp;   
        }
        else  {
            c = gsb[nc * s + n] - 1;  /* use combination for k=0 - as good as any */
            htemp = hxytemps[cc * (mm * s + m) + c]; 
            if (htemp < 1e-200) { result = 0; break; }
            pks = expmin(-htemp);  /* Not captured */
        }
        result *= pks;
        if (dead) break;
    }

    /* resighting phase */
    for (s=qq; s<ss; s++)
    { 
        for (k=0; k<kk; k++)
        {
            state = w[nc * (ss * k + s) + n];
            if (state<0) {dead=1; state=-state;}
            c = gsb[nc * (ss * k + s) + n] - 1;  
            if (c >= 0) {       /* skip if this trap not set */
                indx  = cc * (kk * m + k) + c;  
                              
                if (state>0) {
                    result *= gk[indx];
                    if (s>qq) result *= pID;  /* risk of not ID */
                }

                else result *=  1.0 - gk[indx];
                if (result < minp) {result = minp; break;}  /* truncate */
            }
        }
        if (result < minp) {result = minp; break;}      /* truncate */
        if (dead) break;
    }

    return (result);
}  
/*=============================================================*/

void MRsecrloglik (
    int    *detect,      /* detector 0 multi, 1 proximity */
    int    *distrib,     /* distribution 0 Poisson, 1 binomial */
    int    *w,           /* capture histories (1:nc, 1:s, 1:k) */
    int    *Tu,          /* unmarked counts s = qq+1 ... ss, dim=c(*kk, *ss, *gg) */
    int    *Tm,          /* marked not ID counts s = qq+1 ... ss, dim=c(*kk, *ss, *gg) */
    int    *grp,         /* group number for 0<=n<*nc   [full likelihood only] */
    int    *nc,          /* number of capture histories */
    int    *ss,          /* total number of occasions */
    int    *qq,          /* number of marking occasions */
    int    *kk,          /* number of traps */
    int    *mm,          /* number of points on mask */
    int    *gg,          /* number of groups */
    double *traps,       /* x,y locations of traps (first x, then y) */
    double *mask,        /* x,y points on mask (first x, then y) */
    double *Dmask,       /* density at each point on mask, possibly x group */
    double *gsbval,      /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) */
    double *gsb0val,     /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) [naive animal] */
    int    *cc,          /* number of g0/sigma/b combinations  */
    int    *cc0,         /* number of g0/sigma/b combinations for naive animals */
    int    *gsb,         /* lookup which g0/sigma/b combination to use for given n, S, K */
    int    *gsb0,        /* lookup which g0/sigma/b combination to use for given g, S, K [naive animal] */
    double *pID,         /* Parameter value - probability marked animal identified on resighting */
    double *area,        /* area associated with each mask point (ha) */
    int    *fn,          /* code 0 = halfnormal, 1 = hazard rate, 2 = exponential */
    int    *binomN,      /*  */
    double *minprob,     /* minimum value of P(detection history) */
    double *value,       /* return value integral of pr(w0) */
    int    *resultcode   /* 0 if OK */
)

/*
   Input capthist must be 3-dimensional, even if marking phase is 'multi'

   Yet to add 2009 10 08
       multiple groups (now *gg must be 1)
       count proximity detectors for resighting (i.e. accumulated counts)
*/

{
    int    i,n,g,k,m,c,s;
    int    ng[*gg];       /* number per group */
    int    nSK;
    int    notset = 0;
    double temp;
    double sumD[*gg];     /* 2009 08 21  for binomial */
    double sumDp[*gg];
    
    /* resighting mu accumulators */
    double summuskU [*kk * (*ss - *qq)];   /* zero columns trimmed before input */
    double summuskM [*kk * (*ss - *qq)];   /* zero columns trimmed before input */

    double *gk;
    double *gk0;
    double *hxytemps;
    /* double (*gfn) (
        int k, 
        int m, 
        int c, 
        double gsbval[], 
        int cc, 
        double traps[], 
        double mask[], 
        int kk, 
        int mm); */

    double (*gfn) (
        int k, 
        int m, 
        int c, 
        double gsbval[], 
        int cc, 
        double traps[], 
        double mask[], 
        int kk, 
        int mm, 
        double cut,         /* always 0 */
        int spherical);     /* always 0 */


    double (*MRprwfn) (
        int m, 
        int n,
        int w[],
        int gsb[],
        double gk[],
        double pID,
        double hxytemps[],
        int cc, 
        int nc,
        int kk,
        int ss,
        int qq, 
        int mm,
        double minp);

    /*===============================================================*/

    int debug=0;
    int muindex;
    double pkstemp;
    double pnd;
    double DX;

    /* MAINLINE */

    if (debug) {
        out = fopen( "d:\\debug1.txt", "w" );
        fprintf(out, "Start\n");
        fflush(out);
    }

    *resultcode = 1;  /* generic failure code */
                      /* reset to 0 at end */

    gk = malloc(*cc * *kk * *mm * sizeof (double));
    if (gk == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }
    
    gk0 = malloc(*cc * *kk * *mm * sizeof (double));
    if (gk0 == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }

    /*  allocate enough space to allow for the case where some 
        traps are not set */
    hxytemps = malloc (*cc * *mm * *qq * sizeof(double));          /* marking occasions only */
    if (hxytemps == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }


    /* set functions appropriate to options */
    if (*detect == 0) MRprwfn = MRprwimulti; else MRprwfn = MRprwiprox;

    /* future option: count proximity detector MRprwfn = MRprwicount */

    if (*fn == 0) gfn = ghn;  
    else if (*fn == 1) gfn = ghz;
    else if (*fn == 2) gfn = ghe;
    else if (*fn == 5) gfn = ghf;
    else return;

    for (k=0; k<*kk; k++)
      for (m=0; m<*mm; m++) {
          for (c=0; c<*cc0; c++)
              gk0[*cc0 * (*kk * m + k) + c] = gfn(k, m, c, gsb0val, *cc0, 
                  traps, mask, *kk, *mm, 0, 0);
          for (c=0; c<*cc; c++)
              gk[*cc * (*kk * m + k) + c] = gfn(k, m, c, gsbval, *cc, 
                  traps, mask, *kk, *mm, 0, 0);
      }

 
    if (*detect == 0)     /* preliminaries needed only for multicatch traps */
    {
        /* check if any traps not set */
        /* and decide whether to use full or collapsed hxy */
        nSK = *nc * *qq * *kk;
        for (i=0; i<nSK; i++) if (gsb[i]==0) notset++;

        if (notset==0) {
          for (c=0; c<*cc; c++)
            for (m=0; m<*mm; m++) 
            {
              hxytemps[*cc * m + c] = hxy(c,m, gk, *cc, *kk);
              for (s=1; s<*qq; s++)
                  hxytemps[*cc * (*mm * s + m) + c] = hxytemps[*cc * m + c];
            }
        }
        else {    /* some traps not set: occasion-specific total hazard */
          for (s=0; s<*qq; s++)  
            for (c=0; c<*cc; c++)
              for (m=0; m<*mm; m++) 
                hxytemps[*cc * (*mm * s + m) + c] = hxys(c,m,s, gsb, gk, *cc, *nc, *qq, *kk);
        }
    }

    /* assume *like==0  Full likelihood */
if (debug) fprintf (out, " pID %12.6f \n", *pID);             
if (debug) fprintf (out, " area %12.6f \n", *area);             

    for (g=0; g<*gg; g++) {
        ng[g] = 0;  /* zero for later */
        sumD[g] = 0;
        sumDp[g] = 0;
        for (s= 0; s< (*ss-*qq); s++)
            for (k=0; k< *kk; k++) {
                summuskU [*kk * s + k] = 0;
                summuskM [*kk * s + k] = 0;
            }
        for (m=0; m<*mm; m++)  {
            DX = Dmask[*mm * g + m];
            pnd = pndot (m, g, *gg, gsb0, gk0, *qq, *kk, *cc0);
            sumD[g] += DX;
            sumDp[g] += DX * pnd;

             /* also prepare for TU, TM */
            for (s= 0; s< (*ss-*qq); s++)   /* extra loops - will be slow */
            for (k=0; k< *kk; k++) {
                /* Probability animal n caught in trap k from mask point m on occasion s */
                /* must use absolute s ( s + *qq) to access right values */
                c = gsb[*nc * (*ss * k + (s + *qq)) + g] - 1;               /* use g for n? check 2009 10 06*/
                if (c >= 0)  {                                              /* skip if this trap not set */
                    pkstemp = gk[*cc * (*kk * m + k) + c];
                    muindex = *kk * s + k;                                  /* ASSUME ONE GROUP, FOR NOW */
                    summuskU [muindex] += DX * pkstemp *(1 - pnd);          /* sighting, unmarked */
                    summuskM [muindex] += DX * pkstemp * pnd * (1 - *pID);  /* sighting, marked but notID */
                }
            }
        }
    }

    /*******************************/ 
    /************* L_M *************/

    *value = 0;
    for (n=0; n<*nc; n++) /* CH are numbered 0 <= n < *nc in C code */
    {   
        g = grp[n]-1;
        ng[g]++; 
        temp = 0;
        for (m=0; m<*mm; m++) 
        {
            temp += MRprwfn (m, n, w, gsb, gk, *pID, hxytemps, *cc, *nc, *kk, *ss, *qq, *mm, *minprob) * Dmask[*mm * g + m];    
        }
        *value += log(temp);
    }
    if (debug) fprintf(out, "L_M %15.8f \n", *value);

    for (g=0; g<*gg; g++) {
        *value -= ng[g] * log(sumDp[g]);
        /* Poisson */
        if (*distrib==0) *value += ldpois(ng[g], sumDp[g] * *area); 
        /* binomial */
        if (*distrib==1) *value += ldbinom (ng[g], sumDp[g] / sumD[g], sumD[g] * *area);
    }

    if (debug) fprintf(out, "L_M plus %15.8f \n", *value);

    /*************************************/ 
    /************* L_T, L_T' *************/

    for (s= 0; s< (*ss-*qq); s++)
        for (k=0; k< *kk; k++) {
            muindex = *kk *s + k;      /* ASSUME ONE GROUP, FOR NOW */
            *value += ldpois (Tu[muindex], summuskU[muindex] * *area);
            *value += ldpois (Tm[muindex], summuskM[muindex] * *area);
        }
    /*************************************/ 

    if (debug) fprintf(out, "L_T %15.8f \n", *value);

    /* free dynamic memory allocated with malloc */
    free (gk);
    free (gk0);
    free (hxytemps);

    if (debug) {
        fprintf(out, "\n");
        fclose(out);
    }

    *resultcode = 0;   /* successful termination */
}
/*==============================================================================*/

/*
    trappingXXXX routines perform simulated sampling of 2D popn with various 
    detector types
*/

double pfn (
    int fn,
    double d2val, 
    double g0,
    double sigma,
    double z,
    double area)
{
    double p = -1;
    if (fn == 0) p = g0 * expmin(-d2val / 2 / sigma / sigma);
    else if (fn == 1) p = g0 * (1 - expmin(- pow (sqrt(d2val) / sigma, - z)));
    else if (fn == 2) p = g0 * expmin(- sqrt(d2val) / sigma);
    else if (fn == 3) { if (sqrt(d2val) <= sigma) p = g0; else p = 0; }
    else if (fn == 5) { if (sqrt(d2val) <= z) p = g0; else p = g0 * expmin(- (sqrt(d2val)-z) / sigma); }
    return (p);
}

void trappingproximity (
    /* from 2009 10 08 the following are vectors */
    double *g0,        /* Parameter : detection magnitude */
    double *sigma,     /* Parameter : detection scale */   
    double *z,         /* Parameter : detection shape (hazard) */

    int    *ss,        /* number of occasions */
    int    *kk,        /* number of traps */
    int    *N,         /* number of animals */
    double *animals,   /* x,y points of animal range centres (first x, then y) */
    double *traps,     /* x,y locations of traps (first x, then y) */
    int    *fn,        /* code 0 = halfnormal, 1 = hazard, 2 = exponential, 3 = uniform */
    int    *n,         /* number of individuals caught */
    int    *caught,    /* caught in session */
    int    *value,     /* return value matrix of trap locations n x s */
    int    *resultcode
)

{
    int    capt[*N][*ss][*kk];
    double p;
    double d2val;
    int    i,j,k,l,s;
    int    nc;

    *resultcode = 1;
    nc = 0;
    GetRNGstate();

    for (i=0; i<*N; i++)
    {
        caught[i] = 0;
        for (k=0; k<*kk; k++)
        {
            d2val = d2(i,k, animals, traps, *N, *kk);
  
            for (s=0; s<*ss; s++)
            {
                p = pfn(*fn, d2val, g0[s], sigma[s], z[s], 0);  /* occasion-specific */

                if (p>0)
                if (Random() < p)
                {
                     if (caught[i]==0)           /* first capture of this animal */
                     {
                         caught[i] = 1;
                         nc++; 
                         for (j=0; j<*ss; j++)
                           for (l=0; l<*kk; l++)
                             capt[nc-1][j][l] = 0;
                     } 
                     capt[nc-1][s][k] = 1;
                }
            }
        }     
    }
    *n = nc;
    for (i=0; i<*n; i++)
      for (s=0; s<*ss; s++)
        for (k=0; k<*kk; k++)
          value[*n * (k * *ss + s) + i] = capt[i][s][k];
    *resultcode = 0;
    PutRNGstate();
}
/*==============================================================================*/

double randomtime (double p)
/* return random event time for event with probability p */
{
    double minprob = 1e-5;
    double lambda;
    double random_U;

    if (p < minprob)
        return(huge);                        /* ignore trivial p/lambda */
    else if (p >= 1.0)
        return (-Random());                  /* trick to spread P=1 */
    else {
        lambda   = -log(1-p);                /* rate parameter */
        random_U = Random();
        if (random_U <= 0)                   /* trap for zero */
            return(huge);
        else 
            return (-log(random_U)/lambda);   /* random exponential e.g. Ripley 1987 Algorithm 3.2 p 55 */
    }
}
/*==============================================================================*/

    void probsort (int n, struct trap_animal tran[])
    /*
        Sort using Shell algorithm see Press et al 1989 p 257 
        tran is an array of trap_animal records 
    */
    {
       double aln2i = 1.442695022;
       double tiny  = 1.0e-5;
       int nn,m,lognb2,l,k,j,i;
       struct trap_animal t; 
    
       lognb2 = trunc(log(n)*aln2i+tiny);
       m = n;
       for (nn=1; nn<=lognb2; nn++)
       {
          m = m / 2;
          k = n-m;
          for (j=1; j<=k; j++)
          { 
             i = j;
    lab1:    l = i+m;
             if (tran[l-1].time < tran[i-1].time) 
             {
                t = tran[i-1];
                tran[i-1] = tran[l-1];
                tran[l-1] = t;
                i = i-m;
                if (i >= 1)  goto lab1;
             }
          }
       }
    }    /* end of probsort */
/*==============================================================================*/

void simsecr (
    int    *detect,  /* detector 0 multi, 1 proximity, 2 single, 3 count, 4 area */
    double *gsb0val, /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) [naive animal] */
    double *gsb1val, /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) [caught before] */
    int    *cc0,     /* number of g0/sigma/b combinations for naive animals */
    int    *cc1,     /* number of g0/sigma/b combinations for caught before */
    int    *gsb0,    /* lookup which g0/sigma/b combination to use for given g, S, K [naive animal] */
    int    *gsb1,    /* lookup which g0/sigma/b combination to use for given n, S, K [caught before] */
    int    *N,       /* number of animals */
    int    *ss,      /* number of occasions */
    int    *kk,      /* number of traps */
    double *animals, /* x,y points of animal range centres (first x, then y) */
    double *traps,   /* x,y locations of traps (first x, then y) */
    int    *used,    /* ss x kk array of 0/1 codes for usage */
    int    *Markov,  /* code 0 if behavioural response is learned, 1 if Markov */
    int    *binomN,  /* number of trials for 'count' detector modelled with binomial */
    double *area,    /* area of search cell associated with each detector */
    int    *fn,      /* code 0 = halfnormal, 1 = hazard, 2 = exponential, 3 = uniform */
    int    *n,       /* number of individuals caught */
    int    *caught,  /* sequence number in session (0 if not caught) */
    int    *value,   /* return value array of trap locations n x s */
    int    *resultcode
)
{
    int    capt[*N][*ss][*kk];
    double d2val;
    double p;
    int    i,j,k,l,s;
    int    nc;
    int    count = 0;
    int    caughtbefore[*N];

    double h[*N][*kk];      /* multi-catch only */
    double hsum[*N];        /* multi-catch only */
    double cump[*kk+1];     /* multi-catch only */
    double runif;

    int index;
    int c;
    double g0 = 0;
    double sigma = 0;
    double z = 0;

    /*========================================================*/
    /* 'single-catch only' declarations */

    int    tr_an_indx = 0;  
    int    nanimals;        
    int    ntraps;          
    int    occupied[*kk];   
    int    intrap[*N];      
    struct trap_animal tran[*N * *kk];  
    double event_time;
    int    anum;
    int    tnum;
    int    nextcombo;
    int    finished;
    int    OK;

    /* end of single-only declarations */
    /*========================================================*/

    int debug=0;
    if (debug) {
        out = fopen( "d:\\debug1.txt", "w" );
        fprintf(out, "Start\n");
        fflush(out);
    }

   
    /* MAIN LINE */

    *resultcode = 1;
    cump[0] = 0;     /* multi-catch only */
    nc = 0;
    for (i=0; i<*N; i++) caught[i] = 0;
    GetRNGstate();

    for (s=0; s<*ss; s++)
    {
        for (i=0; i<*N; i++)
        {
            if ((s>0) & *Markov) {
                caughtbefore[i] = 0;
                for (k=0;k<*kk;k++)
                    if(capt[caught[i]-1][s-1][k]>0) caughtbefore[i] = caught[i];
            }
            else
                caughtbefore[i] = caught[i];
        }

        /* multi-catch trap; only one site per occasion (drop last dimension of capt) */
        if (*detect == 0) {  
            for (i=0; i<*N; i++)
            {

                hsum[i] = 0;
                for (k=0; k<*kk; k++) {
                    index = k * (*N * *ss) + s * *N + i;
                    if (caughtbefore[i]) {
                        c = gsb0[index]-1;
                        g0 = gsb0val[c];
                        sigma = gsb0val[*cc0 + c];
                        if ((*fn==1) | (*fn == 5)) z = gsb0val[2* *cc0 + c];
                    }
                    else {                 
                        c = gsb1[index]-1;
                        g0 = gsb1val[c];
                        sigma = gsb1val[*cc1 + c];
                        if ((*fn==1) | (*fn == 5)) z = gsb1val[2* *cc1 + c];
                    }
                    if (*detect == 4) g0 = *area * g0;   /* adjust for area searched */

                    d2val = d2(i,k, animals, traps, *N, *kk);
                    p = pfn(*fn, d2val, g0, sigma, z, 0);         
                    p = p * used[s * *kk + k];           /* zero if not used */
                    h[i][k] = -log(1 - p);
                    hsum[i] += h[i][k];
                }
                for (k=0; k<*kk; k++) {
                    cump[k+1] = cump[k] + h[i][k]/hsum[i];
                }                 
                if (Random() < (1-exp(-hsum[i]))) {
                    if (caught[i]==0)  {        /* first capture of this animal */
                        nc++; 
                        caught[i] = nc;
                        for (j=0; j<*ss; j++)
                            capt[caught[i]-1][j][0] = 0;
                    } 
                    /* find trap with probability proportional to p
                       searches cumulative distribution of p  */
                    runif = Random();
                    k = 0;
                    while ((runif > cump[k]) & (k<*kk)) k++;
                    capt[caught[i]-1][s][0] = k;
                }
            }

/*
    if (debug) {
        for (i=0; i<*N; i++) fprintf(out, "i %10d  s %10d  trap %10d \n", i, s, intrap[i]);
        fflush(out);
    }
*/

        }
        /* single-catch traps */
        else if (*detect == 2) {  
            for (s=0; s<*ss; s++)
            {
                /* initialise day */
                tr_an_indx = 0;
                nanimals = *N;
                ntraps   = *kk;
                for (i=0; i<*N; i++) intrap[i] = 0;
                for (k=0; k<*kk; k++) occupied[k] = 0;
                nextcombo = 0;

                /* make tran */
                for (i=0; i<*N; i++)   /* animals */
                for (k=0; k<*kk; k++)  /* traps */
                {
                    index = k * (*N * *ss) + s * *N + i;
                    if (caughtbefore[i]) {
                        c = gsb0[index]-1;
                        g0 = gsb0val[c];
                        sigma = gsb0val[*cc0 + c];
                        if ((*fn==1) | (*fn == 5)) z = gsb0val[2* *cc0 + c];
                    }
                    else {                 
                        c = gsb1[index]-1;
                        g0 = gsb1val[c];
                        sigma = gsb1val[*cc1 + c];
                        if ((*fn==1) | (*fn == 5)) z = gsb1val[2* *cc1 + c];
                    }
        
                    /* review following... */
                    if (*detect == 4) g0 = *area * g0;   /* adjust for area searched */
                 
                    d2val = d2(i,k, animals, traps, *N, *kk);
                    p = pfn(*fn, d2val, g0, sigma, z, 0);         
                    p = p * used[s * *kk + k];  /* zero if not used */
                    event_time = randomtime(p);
                    if (event_time <= 1)
                    {
                        tran[tr_an_indx].time   = event_time;
                        tran[tr_an_indx].animal = i;    /* 0..*N-1 */
                        tran[tr_an_indx].trap   = k;    /* 0..*kk-1 */
                        tr_an_indx++;
                    }
                }
                /* end of make tran */

                if (tr_an_indx > 1) probsort (tr_an_indx, tran);        

                while ((nextcombo < tr_an_indx) & (nanimals>0) & (ntraps>0)) 
                {
                    finished = 0;
                    OK       = 0; 
                    while ((1-finished)*(1-OK) > 0)  /* until finished or OK */
                    {
                        if (nextcombo >= (tr_an_indx)) finished = 1;  /* no more to process */
                        else 
                        {
                            anum = tran[nextcombo].animal;
                            tnum = tran[nextcombo].trap;
                            OK = (1-occupied[tnum]) * (1-intrap[anum]); /* not occupied and not intrap */
                            nextcombo++;       
                        }
                    }
              
                    if (finished==0)
                    {
                           /* Record this capture */
                          occupied[tnum] = 1;
                          intrap[anum]   = tnum+1;  /* trap = k+1 */
                          nanimals--;
                          ntraps--;
                    }
                }

                for (i=0; i<*N; i++)
                if (intrap[i]>0)
                {
                    if (caught[i]==0)           /* first capture of this animal */
                     {
                       nc++; 
                       caught[i] = nc;          /* nc-th animal to be captured */
                       for (j=0; j<*ss; j++) 
                           capt[caught[i]-1][j][0] = 0;
                     } 
                     capt[caught[i]-1][s][0] = intrap[i];  /* trap = k+1 */
                } 
            }
        }

        /* the 'proximity' group of detectors - proximity, area search, count */
        else   
        for (i=0; i<*N; i++)
        {
            for (k=0; k<*kk; k++)
            {
                index = k * (*N * *ss) + s * *N + i;
                if (caughtbefore[i]) {
                    c = gsb0[index]-1;
                    g0 = gsb0val[c];
                    sigma = gsb0val[*cc0 + c];
                    if ((*fn==1) | (*fn == 5)) z = gsb0val[2* *cc0 + c];
                }
                else {                 
                    c = gsb1[index]-1;
                    g0 = gsb1val[c];
                    sigma = gsb1val[*cc1 + c];
                    if ((*fn==1) | (*fn == 5)) z = gsb1val[2* *cc1 + c];
                }
                if (*detect == 4) g0 = *area * g0;   /* adjust for area searched */

                d2val = d2(i,k, animals, traps, *N, *kk);
                p = pfn(*fn, d2val, g0, sigma, z, 0);         
                p = p * used[s * *kk + k];  /* zero if not used */

                if (p < -0.1) { PutRNGstate(); return; }   /* error */
                if (p>0) {
                    if (*detect == 3) {
                        if (*binomN == 0)
                            count = rpois(p);
                        else
                            count = rbinom(p, *binomN);   /* understand lambdag as probability */
                    }
                    else if (*detect == 1) {
                        count = Random() < p;
                    }

                    if (count>0) {
                        if (caught[i]==0) {          /* first capture of this animal */
                            nc++; 
                            caught[i] = nc;
                            for (j=0; j<*ss; j++)
                              for (l=0; l<*kk; l++)
                                capt[caught[i]-1][j][l] = 0;
                        } 
                        capt[caught[i]-1][s][k] = count;
                    }
                }
            }
        }     
    }
    *n = nc;
    if ((*detect == 0) | (*detect == 2)) *kk = 1;   /* dummy for trapping data (detection scored against k=0)  */
    for (i=0; i<*n; i++)
      for (s=0; s<*ss; s++)
        for (k=0; k<*kk; k++)
          value[*n * (k * *ss + s) + i] = capt[i][s][k];

    PutRNGstate();

    *resultcode = 0;

    if (debug) {
        fprintf(out, "\n");
        fclose(out);
    }

}

/*==============================================================================*/

void trappingarea (
    double *g0,       /* Parameter : detection magnitude */
    double *sigma,    /* Parameter : detection scale */
    double *z,        /* Parameter : detection shape (hazard) */
    int    *ss,       /* number of occasions */
    int    *kk,       /* number of traps */
    int    *N,        /* number of animals */
    double *animals,  /* x,y points of animal range centres (first x, then y)  */
    double *traps,    /* x,y locations of search cell centres (first x, then y)  */
    double *area,     /* area of search cell associated with each point */
    int    *fn,       /* code 0 = halfnormal, 1 = hazard, 2 = exponential */
    int    *n,        /* number of individuals caught */
    int    *caught,   /* caught in session */
    int    *value,    /* return value matrix of trap locations n x s */
    int    *resultcode
)

{
    int    capt[*N][*ss][*kk];
    double d2val;
    double p;
    int    i,j,k,l,s;
    int    nc;

    *resultcode = 1;
    nc = 0;
    GetRNGstate();

    for (i=0; i<*N; i++)
    {
        caught[i] = 0;
        for (k=0; k<*kk; k++)
        {
            d2val = d2(i,k, animals, traps, *N, *kk);
  
            for (s=0; s<*ss; s++)
            {
                p = pfn(*fn, d2val, g0[s], sigma[s], z[s], 0);  /* occasion-specific */

                if (Random() < p)
                {
                     if (caught[i]==0)           /* first capture of this animal */
                     {
                         caught[i] = 1;
                         nc++; 
                         for (j=0; j<*ss; j++)
                           for (l=0; l<*kk; l++)
                             capt[nc-1][j][l] = 0;
                     } 
                     capt[nc-1][s][k] = 1;
                }
            }
        }     
    }
    *n = nc;
    for (i=0; i<*n; i++)
      for (s=0; s<*ss; s++)
        for (k=0; k<*kk; k++)
          value[*n * (k * *ss + s) + i] = capt[i][s][k];
    *resultcode = 0;
    PutRNGstate();

}
/*==============================================================================*/

void trappingcount (
    double *g0,       /* Parameter : detection intercept  */
    double *sigma,    /* Parameter : detection scale */
    double *z,        /* Parameter : detection shape (hazard) */
    int    *ss,       /* number of occasions */
    int    *kk,       /* number of traps */
    int    *N,        /* number of animals */
    double *animals,  /* x,y points of animal range centres (first x, then y)  */
    double *traps,    /* x,y locations of traps (first x, then y)  */
    int    *fn,       /* code 0 = halfnormal, 1 = hazard, 2 = exponential */
    int    *binomN,   /* number of trials for 'count' detector modelled with binomial */
    int    *n,        /* number of individuals caught */
    int    *caught,   /* caught in session */
    int    *value,    /* return value matrix of trap locations n x s */
    int    *resultcode
)

{

    int    capt[*N][*ss][*kk];
    double d2val;
    double lambdag;
    int    i,j,k,l,s;
    int    nc;
    int    count;

    *resultcode = 1;
    nc = 0;
    GetRNGstate();

    for (i=0; i<*N; i++)
    {
        caught[i] = 0;
        for (k=0; k<*kk; k++)
        {
            d2val = d2(i,k, animals, traps, *N, *kk);

            for (s=0; s<*ss; s++)
            {
                lambdag = pfn(*fn, d2val, g0[s], sigma[s], z[s], 0);
                if (lambdag>0) {
                    if (*binomN == 0)
                        count = rpois(lambdag);
                    else
                        count = rbinom(lambdag, *binomN);   /* understand lambdag as probability */
                    if (count>0)
                    {
                         if (caught[i]==0)           /* first capture of this animal */
                         {
                             caught[i] = 1;
                             nc++; 
                             for (j=0; j<*ss; j++)
                               for (l=0; l<*kk; l++)
                                 capt[nc-1][j][l] = 0;
                         } 
                         capt[nc-1][s][k] = count;
                    }
                }
            }
        }     
    }
    *n = nc;
    for (i=0; i<*n; i++)
      for (s=0; s<*ss; s++)
        for (k=0; k<*kk; k++)
          value[*n * (k * *ss + s) + i] = capt[i][s][k];
    *resultcode = 0;
    PutRNGstate();

}
/*==============================================================================*/

void trappingsignal (
    double *beta0,    /* Parameter : intercept */
    double *beta1,    /* Parameter : slope */
    double *sdS,      /* Parameter : error sd */
    double *cut,      /* detection threshold on transformed scale */
    int    *ss,       /* number of occasions */
    int    *kk,       /* number of traps */
    int    *N,        /* number of animals */
    double *animals,  /* x,y points of animal range centres (first x, then y)  */
    double *traps,    /* x,y locations of traps (first x, then y)  */
    int    *fn,       /* code 10 = signal strength, 11 = binary signal strength */
    int    *spherical,
    int    *n,        /* number of individuals caught */
    int    *caught,   /* caught in session */
    double *value,    /* return value matrix of trap locations n x s */
    int    *resultcode
)

/* returned signal strength (*fn==10) is on transformed scale */
{
    double capt[*N][*ss][*kk];
    double mu, signal;
    int    i,j,k,l,s;
    int    nc = 0;
    double zero = 0.0;

    int debug=0;

    /* MAINLINE */

    if (debug) {
        out = fopen( "d:\\debug1.txt", "w" );
        fprintf(out, "Start\n");
        fflush(out);
    }

    *resultcode = 1;
    GetRNGstate();

    /* large neg value = nondetection */
    if ( *fn == 10 ) zero = -1e20;

    for (i=0; i<*N; i++)
    {
        caught[i] = 0;
        for (k=0; k<*kk; k++)
        {
            for (s=0; s<*ss; s++)
            {
                mu  = mufn (i, k, beta0[s], beta1[s], *spherical, animals, traps, *N, *kk);
                signal = norm_rand() * sdS[s] + mu;

                if (signal > *cut) {
                    if (caught[i]==0)           /* first capture of this animal */
                    {
                        caught[i] = 1;
                        nc++; 
                        for (j=0; j<*ss; j++)
                          for (l=0; l<*kk; l++)
                            capt[nc-1][j][l] = zero; 
                    } 
                    if ( *fn == 10 )
                        capt[nc-1][s][k] = signal;
                    else capt[nc-1][s][k] = 1;
                } 
            }
        }     
    }
    *n = nc;
  
    /* copy results - could be avoided */
    for (i=0; i<*n; i++)
      for (s=0; s<*ss; s++)
        for (k=0; k<*kk; k++) 
          value[*n * (k * *ss + s) + i] = capt[i][s][k];
    *resultcode = 0;

    PutRNGstate();

    if (debug) {
        fprintf(out, "\n");
        fclose(out);
    }


}
/*==============================================================================*/

void trappingmulti (
    double *g0,         /* Parameter : detection magnitude  */
    double *sigma,      /* Parameter : detection scale */
    double *z,          /* Parameter : detection shape (hazard) */
    int    *ss,         /* number of occasions */
    int    *kk,         /* number of traps */
    int    *N,          /* number of animals */
    double *animals,    /* x,y points of animal range centres (first x, then y)  */
    double *traps,      /* x,y locations of traps (first x, then y)  */
    int    *fn,         /* code 0 = halfnormal, 1 = hazard, 2 = exponential */
    int    *n,          /* number of individuals caught */
    int    *caught,     /* caught in session */
    int    *value,      /* return value matrix of trap locations n x s */
    int    *resultcode  /* 0 for successful completion */
)
  
{
    int    capt[*N][*ss];
    double h[*N][*kk];
    double hsum[*N];
    double cump[*kk+1];
    double runif;
    int    i,j,k,s;
    int    nc;
    double d2val;
    double p;
  
    *resultcode = 1;
    cump[0] = 0;
    nc = 0;
    GetRNGstate();
  
    for (i=0; i<*N; i++)
    {

        caught[i] = 0;
        for (s=0; s<*ss; s++)
        {
            hsum[i] = 0;
            for (k=0; k<*kk; k++)
            {
                d2val = d2(i,k, animals, traps, *N, *kk);
                p = pfn(*fn, d2val, g0[s], sigma[s], z[s], 0);  
                h[i][k] = -log(1 - p);
                hsum[i] += h[i][k];
            }
            for (k=0; k<*kk; k++) 
            {
                cump[k+1] = cump[k] + h[i][k]/hsum[i];
            }

            if (Random() < (1-exp(-hsum[i])))
            {
               if (caught[i]==0)           /* first capture of this animal */
               {
                   caught[i] = 1;
                   nc++; 
                   for (j=0; j<*ss; j++) capt[nc-1][j] = 0;
               } 
               runif = Random();
               k = 0;
               while ((runif > cump[k]) & (k<*kk)) k++;
               capt[nc-1][s] = k;   /* pick a trap */
            }
        } 
    }
    *n = nc;
    for (i=0; i<*n; i++)
        for (s=0; s<*ss; s++)
            value[i + *n * s] = capt[i][s];
    *resultcode = 0;
    PutRNGstate();

}
/*==============================================================================*/

void trappingsingle (
    double *g0,        /* Parameter : detection magnitude  */
    double *sigma,     /* Parameter : detection scale */
    double *z,         /* Parameter : detection shape (hazard) */
    int    *ss,        /* number of occasions */
    int    *kk,        /* number of traps */
    int    *N,         /* number of animals */
    double *animals,   /* x,y points of animal range centres (first x, then y)  */
    double *traps,     /* x,y locations of traps (first x, then y)  */
    int    *fn,        /* code 0 = halfnormal, 1 = hazard, 2 = exponential, 3 uniform */
    int    *n,         /* number of individuals caught */
    int    *caught,    /* caught in session */
    int    *value,     /* return value matrix of trap locations n x s */
    int    *resultcode /* 0 for successful completion */
)
{
    int    capt[*N][*ss];
    int    i,j,k,s;
    int    nc         = 0;
    int    tr_an_indx = 0;
    double d2val;
    double p;
    int nanimals;         /* temporary */
    int ntraps;           /* temporary */
    int occupied[*kk];    /* today */
    int intrap[*N];       /* today   */
  
    struct  trap_animal tran[*N * *kk];  
    double event_time;
    int anum;
    int tnum;
    int nextcombo;
    int finished;
    int OK;

    /* MAIN LINE */

    *resultcode = 1;
    GetRNGstate();

    for (i=0; i<*N; i++) caught[i] = 0;   /* has animal i been caught in session? */
    nc = 0;

    for (s=0; s<*ss; s++)
    {
        /* initialise day */
        tr_an_indx = 0;
        nanimals = *N;
        ntraps   = *kk;
        for (i=0; i<*N; i++) intrap[i] = 0;
        for (k=0; k<*kk; k++) occupied[k] = 0;
        nextcombo = 0;

        /* make tran */
        for (i=0; i<*N; i++)   /* animals */
        for (k=0; k<*kk; k++)  /* traps */
        {
            d2val = d2(i,k, animals, traps, *N, *kk);
            p = pfn(*fn, d2val, g0[s], sigma[s], z[s], 0);         

            event_time = randomtime(p);
            if (event_time <= 1)
            {
                tran[tr_an_indx].time   = event_time;
                tran[tr_an_indx].animal = i;    /* 0..*N-1 */
                tran[tr_an_indx].trap   = k;    /* 0..*kk-1 */
                tr_an_indx++;
            }
        }

        if (tr_an_indx>0) probsort (tr_an_indx, tran);

        /* make captures */
        while ((nextcombo < tr_an_indx) & (nanimals>0) & (ntraps>0)) 
        {
            finished = 0;
            OK       = 0; 
            while ((1-finished)*(1-OK) > 0)  /* until finished or OK */
            {
                if (nextcombo >= (tr_an_indx)) finished = 1;  /* no more to process */
                else 
                {
                    anum = tran[nextcombo].animal;
                    tnum = tran[nextcombo].trap;
                    OK = (1-occupied[tnum]) * (1-intrap[anum]); /* not occupied and not intrap */
                    nextcombo++;       
                }
            }
      
            if (finished==0)
            {
                   /* Record this capture */
                  occupied[tnum] = 1;
                  intrap[anum]   = tnum+1;  /* trap = k+1 */
                  nanimals--;
                  ntraps--;
            }
        }

        for (i=0; i<*N; i++)
        if (intrap[i]>0)
        {
            if (caught[i]==0)           /* first capture of this animal */
             {
               nc++; 
               caught[i] = nc;          /* nc-th animal to be captured */
               for (j=0; j<*ss; j++) capt[nc-1][j] = 0;
             } 
             capt[caught[i]-1][s] = intrap[i];  /* trap = k+1 */
        } 
    }

    *n = nc;
    for (i=0; i<*n; i++)
        for (s=0; s<*ss; s++)
            value[i + *n * s] = capt[i][s];

    *resultcode = 0;
    PutRNGstate();

}
/*==============================================================================*/

/*
    'naive' functions are used to estimate auto initial values
    these use only the halfnormal detection function 4/5/08
*/

void naived (
  double *sigma,   /* Parameter : detection scale */
  int    *kk,      /* number of traps */
  int    *nc,
  double *traps,   /* x,y locations of traps (first x, then y) */
  double *animals, /* x,y locations of traps (first x, then y) */
  int    *fn,      /* code 0 = halfnormal ONLY */
  double *value    /* return value */
)
{
  double truncate2 = (2.45 * *sigma) * (2.45 * *sigma);
  double sump  = 0;
  double sumdp = 0;
  double x,y;
  double dij, d21, d22, p1p2;
  int i,j,n;

  for (n=0; n<*nc; n++)
  {
    x = animals[n];
    y = animals[n + *nc];

    for (i=0; i<*kk; i++)
      for (j=0; j<(i-1); j++)
        {

        dij = (traps[i] - traps[j]) * (traps[i] - traps[j]) + 
                (traps[i+*kk] - traps[j+*kk]) * (traps[i+*kk] - traps[j+*kk]);
        d21 = (traps[i] - x) * (traps[i] - x) + (traps[i+*kk] - y) * (traps[i+*kk] - y);
        d22 = (traps[j] - x) * (traps[j] - x) + (traps[j+*kk] - y) * (traps[j+*kk] - y);

        if ((d21<=truncate2) & (d22<=truncate2)) 
           p1p2 = exp(-(d21+d22) / 2 / *sigma / *sigma);
        else
           p1p2 = 0;

        sump  += p1p2;
        sumdp += p1p2 * sqrt(dij);
  
        }
    for (i=0; i<*kk; i++)  /* diagonal */
      {
        d21 = (traps[i] - x) * (traps[i] - x) + (traps[i+*kk] - y) * (traps[i+*kk] - y);
        if (d21<=truncate2)                                     /* d21=d22 */
          sump += exp(-2*d21 /2 / *sigma / *sigma)/2;
      }
  }
  *value = sumdp/sump;
}
/*==============================================================================*/

void naivecap2 (
  int    *detect,  /* code 0 = multicatch, 1 = proximity */
  double *g0,      /* Parameter : detection magnitude */
  double *sigma,   /* Parameter : detection scale */
  int    *ss,      /* number of occasions */
  int    *kk,      /* number of traps */
  int    *mm,
  double *traps,   /* x,y locations of traps (first x, then y)  */
  double *mask,    /* x,y points on mask (first x, then y)  */
  int    *fn,      /* code 0 = halfnormal ONLY */
  double *value    /* return value  */
)
{
    double product;
    double d2val;
    double pk;
    int m,k;
    double nsum = 0;
    double psum = 0;
  
    for (m=0; m<*mm; m++)
    {
        product = 1.0;
        for (k=0; k<*kk; k++)
        {
            d2val = d2(m, k, mask, traps, *mm, *kk);
            pk = *g0 * expmin(-d2val / 2 / *sigma / *sigma);
            product *= (1 - pk);
            if (*detect == 1) nsum += pk;
        }      
        if (*detect == 0) nsum += (1 - product);
        psum += 1 - pow(product, *ss);
    }
    if (psum<=0)
      *value = 0;    /* failed */
    else
      *value = *ss * nsum / psum;
}
/*==============================================================================*/

void makelookup (   
  double *x,            /* input matrix */
  int    *nrow,         /* input */
  int    *ncol,         /* input */
  int    *unique,       /* output number of unique rows */
  double *y,            /* output matrix of unique rows (byrow=T) */
  int    *index,        /* output lookup rows of x in y */
  int    *resultcode)   /* zero if OK */

/*
   Create lookup table to the unique rows in a matrix
   Only the first 'unique' rows of y contain valid data on exit
   The indices are 1..length(unique(x))=nrow(y')
   MGE 2008 05 07
*/

{
    int i;    
    int j;
    int k;
    int dupl = 0;
    double *ytemp;

    *resultcode = 1;

    ytemp = malloc(*nrow * *ncol * sizeof (double));
    if (ytemp == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }

    /*
        avoid sort for now as it's complex to keep order of first occurrence, not needed
        scan for unique rows of x, copying to y
        assign unique index to original rows as we go    
    */

    for (j=0; j < *ncol; j++) ytemp[j] = x[*nrow * j];    /* first row */
    index[0] = 1;
    *unique=0;

    for (i=1; i < *nrow; i++) {
       /* Is this row unique? Compare with each previous unique in turn */
       for (k=0; k <= *unique; k++) {
           dupl = 1;
           for (j=0; j < *ncol; j++) {
               if (x[*nrow * j + i] != ytemp[k * *ncol + j]) 
               {dupl=0; break;}
           }
           if (dupl==1) break;  /* found previous instance */
       }
       if (dupl==0) { /* add unique row */
           *unique = *unique + 1;
           k = *unique;
           for (j=0; j< *ncol; j++) ytemp[k * *ncol + j] = x[*nrow * j + i];          
       }
       index[i] = k+1;
    }

    *unique = *unique + 1;   /* number of unique rows */
    for (i=0; i<(*unique * *ncol); i++) y[i] = ytemp[i];

    free (ytemp);
    *resultcode = 0;

}
/*==============================================================================*/

void nearest (   
  double *xy,       /* input point */
  int    *ntrap,    /* input */
  double *traps,    /* input */
  int    *p,        /* output index of nearest point */
  double *d)        /* output distance to nearest point */
{
    int i;
    int id=-1;
    double d2;
    *d = 1e100;
    for (i=0; i<*ntrap; i++) 
    {
        d2 = (traps[i] - xy[0]) * (traps[i] - xy[0]) +
             (traps[i + *ntrap] - xy[1]) * (traps[i + *ntrap] - xy[1]);
        if (d2 < *d) { *d = d2; id = i; }
    }
    *d = sqrt(*d);  
    *p = id+1;
}
/*==============================================================================*/

void inside (
    double *xy, 
    int    *np, 
    double *poly, 
    int    *in)
{
/*
    Is point xy inside poly?
    Based on contribution on s-news list by Peter Perkins 23/7/96
    We assume poly is closed, and in col-major order (x's then y's)
*/
  
    double theta = 0;
    double cutoff = 1e-6;
    int k;
    double N;
    double d;
    
    /* get & translate to coords centered at each test point */
    for (k=0; k < *np; k++) 
    {
        poly[k]     = poly[k] - xy[0];           /* x */
        poly[k + *np] = poly[k + *np] - xy[1];   /* y */
    }
  
    for (k=0; k < (*np-1); k++) 
    {
        N = poly[k] * poly[k+1 + *np] - poly[k + *np] * poly[k+1];
        d = poly[k] * poly[k+1] + poly[k + *np] * poly[k+1 + *np];  
        if (abs(d)>0) { N = N/abs(d);  d = d/abs(d); }
        theta += atan2(N, d);
     }
     theta = abs(theta);
     if (abs(theta - 2* M_PI) < cutoff)    /* M_PI is Rmath.h constant */
         *in = 1;    /* inside */
     else
         *in = 0;    /* outside */
}
/*==============================================================================*/


/* return conditional probability of each capture history */

/*=============================================================*/
/* THIS FUNCTION NOT USED 2009 10 17 */
double prfn (
        int detect,
        int m, 
        int n, 
        int s, 
        int k, 
        int c, 
        double gk[],
        double hxytemps[],
        int cc, 
        int nc,
        int kk,
        int ss,
        int mm)

/*
    Probability of capture at s,k
    given that animal's range centre is at m
*/
{
    int indx;
    double htemp;
    double pks;
    if (detect == 0) {        /* MULTI-CATCH DETECTOR */
        indx = cc * (kk * m + (k-1)) + c;
        htemp = hxytemps[cc * (mm * s + m) + c];
        if (htemp < 1e-200) {
            return(0);
        }
        else {  
            pks = -log (1 - gk[indx]);
            pks *= (1-expmin(-htemp)) / htemp;   
            return(pks);
        }
    }
    else {                    /* PROXIMITY DETECTOR */
        if (c >= 0) {       /* skip if this trap not set */
            indx  = cc * (kk * m + k) + c;                                
            return ( gk[indx] );
        }
        else return (0);
    }

}  
/*=============================================================*/

void secrcellprob (
    int    *detect,      /* detector 0 multi, 1 proximity */
    int    *w,           /* capture histories (1:nc, 1:s, 1:k) */
    int    *grp,         /* group number for 0<=n<*nc   [full likelihood only] */
    int    *nc,          /* number of capture histories */
    int    *ss,          /* number of occasions */
    int    *kk,          /* number of traps */
    int    *mm,          /* number of points on mask */
    int    *gg,          /* number of groups */
    double *traps,       /* x,y locations of traps (first x, then y) */
    double *mask,        /* x,y points on mask (first x, then y) */
    double *Dmask,       /* density at each point on mask, possibly x group */
    double *gsbval,      /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) */
    double *gsb0val,     /* Parameter values (matrix nr= comb of g0,sigma,b nc=3) [naive animal] */
    int    *cc,          /* number of g0/sigma/b combinations  */
    int    *cc0,         /* number of g0/sigma/b combinations for naive animals */
    int    *gsb,         /* lookup which g0/sigma/b combination to use for given n, S, K */
    int    *gsb0,        /* lookup which g0/sigma/b combination to use for given g, S, K [naive animal] */
    double *area,        /* area associated with each mask point (ha) */
    int    *fn,          /* code 0 = halfnormal, 1 = hazard, 2 = exponential */
    int    *binomN,      /* number of trials for 'count' detector modelled with binomial */
    double *cut,         /* transformed signal strength threshold for detection */
    int    *spherical,
    double *minprob,     
    double *value,       /* return value integral of pr(w0) */
    int    *resultcode   /* 0 if OK */
)

{
    int    i,n,g,k,m,c,s;
    int    ng[*gg];      
    int    nSK;
    int    notset = 0;
    double sumD[*gg];    
    double sumDp[*gg];
    double *gk;
    double *gk0;
    double *hxytemps;
    double (*gfn) (
        int k, 
        int m, 
        int c, 
        double gsbval[], 
        int cc, 
        double traps[], 
        double mask[], 
        int kk, 
        int mm, 
        double cut, 
        int spherical);
    double (*prwfn) (
        int m, 
        int n,
        int w[],
        int gsb[],
        double gk[],
        double hxytemps[],
        int cc, 
        int nc,
        int kk,
        int ss,
        int mm,
        double minp);

    /*===============================================================*/

    *resultcode = 1;  /* generic failure code */
                      /* reset to 0 at end */

    gk = malloc(*cc * *kk * *mm * sizeof (double));
    if (gk == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }
    
    gk0 = malloc(*cc * *kk * *mm * sizeof (double));
    if (gk0 == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }

    /*  allocate enough space to allow for the case where some 
        traps are not set */
    hxytemps = malloc (*cc * *mm * *ss * sizeof(double));
    if (hxytemps == NULL)   /* Memory could not be allocated */
    { *resultcode = 2;  return; }

    /* set functions appropriate to options */
    if (*detect == 0) prwfn = prwimulti; else prwfn = prwiprox;

    if (*fn == 0) gfn = ghn;  
    else if (*fn == 1) gfn = ghz;
    else if (*fn == 2) gfn = ghe;
    else if (*fn == 5) gfn = ghf;
    else return;

    for (k=0; k<*kk; k++)
      for (m=0; m<*mm; m++) {
          for (c=0; c<*cc0; c++)
              gk0[*cc0 * (*kk * m + k) + c] = gfn(k, m, c, gsb0val, *cc0, 
                  traps, mask, *kk, *mm, *cut, *spherical);
          for (c=0; c<*cc; c++)
              gk[*cc * (*kk * m + k) + c] = gfn(k, m, c, gsbval, *cc, 
                  traps, mask, *kk, *mm, *cut, *spherical);
      }

    if (*detect == 0)                           /* only for multicatch traps */
    {
        /* check if any traps not set */
        /* and decide whether to use full or collapsed hxy */
        nSK = *nc * *ss * *kk;
        for (i=0; i<nSK; i++) if (gsb[i]==0) notset++;

        if (notset==0) {
          for (c=0; c<*cc; c++)
            for (m=0; m<*mm; m++) 
            {
              hxytemps[*cc * m + c] = hxy(c,m, gk, *cc, *kk);
              for (s=1; s<*ss; s++)
                  hxytemps[*cc * (*mm * s + m) + c] = hxytemps[*cc * m + c];
            }
        }
        else {    /* some traps not set: occasion-specific total hazard */
          for (s=0; s<*ss; s++)  
            for (c=0; c<*cc; c++)
              for (m=0; m<*mm; m++) 
                hxytemps[*cc * (*mm * s + m) + c] = hxys(c,m,s, gsb, gk, *cc, *nc, *ss, *kk);
        }
    }

    /* Assume full likelihood */
        for (g=0; g<*gg; g++) {
            ng[g] = 0;  /* zero for later */
            sumD[g] = 0;
            sumDp[g] = 0;
            for (m=0; m<*mm; m++)  {
                sumD[g] += Dmask[*mm * g + m];
                sumDp[g] += Dmask[*mm * g + m] * pndot (m, g, *gg, gsb0, gk0, *ss, *kk, *cc0);
            }
        }
 
        for (n=0; n<*nc; n++) /* CH are numbered 0 <= n < *nc in C code */
        {   
            g = grp[n]-1;
            ng[g]++; 
            value[n] = 0;
            for (m=0; m<*mm; m++) 
            {
                value[n] += prwfn (m, n, w, gsb, gk, hxytemps, *cc, *nc, *kk, *ss, *mm, *minprob)
                   * Dmask[*mm * g + m];    
            }
            value[n] = value[n] / (sumDp[g] * *area);
        }

    /* free dynamic memory allocated with malloc */
    free (gk);
    free (gk0);
    free (hxytemps);

    *resultcode = 0;   /* successful termination */
}
/*==============================================================================*/


