\name{esa.plot}
\alias{esa.plot}
\alias{esa.plot.secr}

\title{
Mask Buffer Diagnostic Plot
}
\description{
Plot effective sampling area (Borchers and Efford 2008) as a function of increasing buffer width.
}

\usage{

esa.plot (traps, max.mask, detectfn, detectpar, noccasions,
    thin = 0.1, plt = TRUE, invert = TRUE, n = 1, add = FALSE, ...)

esa.plot.secr (object, max.buffer = NULL, max.mask = NULL, thin = 0.1,
    plt = TRUE, invert = TRUE, add = FALSE, ...)

}

\arguments{
  \item{traps}{ \code{traps} object }
  \item{max.mask}{ \code{\link{mask}} object }
  \item{detectfn}{ integer code or character string for shape of
  detection function 0 = halfnormal etc. -- see \link{detectfn}}
  \item{detectpar}{ list of values for named parameters of detection function }
  \item{noccasions}{ number of sampling occasions}
  \item{thin}{ proportion of mask points to retain in plot and output}
  \item{plt}{ logical to plot results }
  \item{invert}{ logical; if TRUE the y-axis is n / esa }
  \item{n}{ integer number of distinct individuals detected }
  \item{add}{ logical to add line to an existing plot }
  \item{\dots}{ graphical arguments passed to plot() and lines()}
  \item{object}{ \code{secr} object output from \code{secr.fit} }
  \item{max.buffer}{ maximum width of buffer in metres }
}

\details{

  Effective sampling area (esa) is defined as the integral of net
  capture probability (\eqn{p.(\mathbf{X})}{p.(X)}) over a
  region. \code{esa.plot} shows the effect of increasing region size on
  the value of esa for fixed values of the detection parameters. The
  \code{max.mask} argument establishes the maximum extent of the region;
  points (cells) within this mask are sorted by their distance
  \eqn{d_k}{dk} from the nearest detector. esa(buffer) is defined as the
  cumulative sum of \eqn{cp.(\mathbf{X})}{c * p.(X)} for
  \eqn{d_k(\mathbf{X}) <= \mbox{buffer}}{dk(X) <= buffer}, where \eqn{c}
  is the area associated with each cell.

  The default (\code{invert = TRUE}) is to plot the reciprocal of esa
  multiplied by \code{n}; this is on a more familiar scale (the density
  scale) and hence is more directly interpretable.
  
  Because \code{esa.plot} uses the criterion `distance to nearest
  detector', \code{max.mask} should be constructed to include all
  habitable cells within the desired maximum buffer and no others. This
  is achieved with \code{type = 'trapbuffer'} in \code{make.mask}. It is
  a good idea to set the \code{spacing} argument of \code{make.mask}
  rather than relying on the default based on \code{nx}. Spacing may be
  small (e.g. sigma/10) and the buffer of \code{max.mask} may be quite
  large (e.g. 10 sigma), as computation is fast.
  
  Thinning serves to reduce redundancy in the plotted points, and (if
  the result is retained) to generate more legible numerical output. Use
  \code{thin=1} to include all points.

  \code{esa.plot.secr} provides a wrapper for \code{esa.plot} that
  avoids the need to specify arguments separately.

  If \code{max.mask} is not specified in \code{esa.plot.secr} then a
  maximal mask of type `trapbuffer' is constructed using
  \code{max.buffer} and the spacing of the mask in \code{object}. In
  this case, if \code{max.buffer} is not specified then it is set either
  to the width of the existing plot (\code{add = TRUE}) or to 10 x
  sigma-hat from the fitted model in \code{object} (\code{add = FALSE}).
  
}

\value{
  
  A dataframe with columns
  \item{buffer}{buffer width}
  \item{esa}{computed effective sampling area}
  \item{pdotmin}{cumulative minimum  (\eqn{p.(\mathbf{X})}{p.(X)})}
  If \code{plt = TRUE} the dataframe is returned invisibly.

}

\note{
The response of effective sampling area to buffer width is just one
possible mask diagnostic; it's fast, graphic, and often
sufficient. \code{\link{mask.check}} performs more intensive checks,
usually for a smaller number of buffer widths.
}

\references{

Borchers, D. L. and Efford, M. G. (2008) Spatially explicit maximum
likelihood methods for capture--recapture studies. \emph{Biometrics}
\bold{64}, 377--385.

}

\seealso{ \code{\link{mask}}, \code{\link{pdot}}, \code{\link{make.mask}}, \code{\link{mask.check}},
\code{\link{detection functions}} }

\examples{

## with fitted model
data(secrdemo)
esa.plot.secr(secrdemo.0)

## from scratch
trps <- make.grid()
msk <- make.mask(trps, buffer = 200, spacing = 5, type = 'trapbuffer')
detectpar <- list(g0 = 0.2, sigma = 25)
esa.plot(trps, msk, 0, detectpar, nocc = 10, col = 'blue')
esa.plot(trps, msk, 0, detectpar, nocc = 5, col = 'green',
    add = TRUE)

esa.plot(trps, msk, 0, detectpar, nocc = 5, thin = 0.002, plt = FALSE)

}
\keyword{ hplot }
